<?php

namespace App\Http\Controllers;

use App\Client;
use App\ClientToken;
use App\Http\Controllers\Auth\RegisterController;
use App\Jobs\SendConfirmClientAdminEMail;
use App\Mail\ConfirmClientAdmin;
use Illuminate\Http\Request;

use App\Http\Controllers\ClientTokenController;
use View;
use Log;
use App\User;
use URL;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use JsValidator;
use Validator;
use App\Operator;

class ClientController extends Controller {

    private function getRegisterFirstClientAdminValidationRules() {
        $registerController = new RegisterController();
        return $registerController->getValidationRules();
    }

    private function getUpdateValidationRules() {
        return [
            'name' => 'required|string|max:255',
            'street' => 'required|regex:/\b[A-Za-z\.\-\säÄöÖüÜß]+/|max:255',
            'houseNumber' => 'required|regex:/\b[0-9]{1,4}[a-zA-Z]{0,1}\b/',
            'zipcode' => 'required|regex:/\b[0-9]{4,6}\b/',
            'place' => 'required|regex:/\b[A-Za-z.\-\säÄöÖüÜß]+\b/|max:255',
            'country' => 'required|regex:/\b[A-Za-z\-\säÄöÖüÜß]+\b/|max:255',
            'vatNo' => 'required|regex:/\bDE[0-9]{9}\b/'
        ];
    }

    public function showClientAdminCreation($token) {
        # Prüfen, ob Token noch gültig ist
        $tokenEntry = ClientTokenController::checkTokenValidity("register", $token);
        if (!$tokenEntry)
            return View::make('client.clientAdminEdit')->withErrors(["Der Link zur Registrierung eines neuen Mandanten ist nicht mehr gültig."]);

        # Mandantendaten laden
        $client = Client::find($tokenEntry->client_id);
        if (!$client) {
            Log::error("Kürzlich registrierter Mandant " . $tokenEntry->client_ind . " nicht mehr gefunden!");
            return View::make('client.clientAdminEdit')->withErrors(["Es ist ein Fehler aufgetreten"]);
        }

        # Validierung erstellen
        $frontValidator = JsValidator::make($this->getRegisterFirstClientAdminValidationRules(), [], [], "#registerFirstClientAdmin");

        return View::make('client.clientAdminEdit')->with(["client" => $client, "token" => $token, "validator" => $frontValidator]);
    }

    public function showClientCreationErrorPage() {
        return view('outsideIndex');
    }

    # Mandantenadministrator-Daten verarbeiten, registrieren und Bestätigungsmail versenden
    public function registerFirstClientAdmin(Request $request) {
        $token = $request->get("clientToken");
        $tokenEntry = ClientTokenController::checkTokenValidity("register", $token);

        if (!$tokenEntry)
            return redirect()->route('indexOutside')->withErrors(["Der Link zur Registrierung eines neuen Mandanten ist nicht mehr gültig."]);

        # Mandantendaten laden
        $client = Client::find($tokenEntry->client_id);
        if ($client == null) {
            Log::error("Kürzlich registrierter Mandant " . $tokenEntry->client_id . " nicht mehr gefunden!");
            return redirect()->route('indexOutside')->withErrors(["Es ist ein Fehler aufgetreten"]);
        }

        # Formulardaten validieren
        $backValidator = Validator::make(request()->all(), $this->getRegisterFirstClientAdminValidationRules());
        if ($backValidator->fails())
            return redirect()->back()->withErrors($backValidator->errors());

        # Neuen Benutzer registrieren (Mandantenadministrator)
        $user = new User();
        $user->name = $request->get("name");
        $user->firstname = $request->get("firstname");
        $user->email = $request->get("email");
        $user->password = Hash::make($request->get("password"));


        try {
            $user->save();
        } catch (\Exception $e) {
            Log::error("Fehler bei der Neuregistrierung eines Mandantenadministrators für Mandant " . $tokenEntry->client_id .
                "mit E-Mail-Adresse " . $request->get("email"));
            Log::error($e->getMessage());
            return redirect()->route('indexOutside')->withErrors(["Es ist ein Fehler aufgetreten."]);
        }

        # Mandantenadministrator dem neuen Mandanten zuordnen
        try {
            self::addUserToClient($client, $user);
        } catch (\Exception $e) {
            Log::error("Fehler beim Zuordnen eines neuen Administrators zu " . $tokenEntry->client_id .
                " (User" . $user->id);
            Log::error($e->getMessage());
            return redirect()->route('indexOutside')->withErrors(["Es ist ein Fehler aufgetreten."]);
        }

        # Rechte zuweisen
        $rightsController = new RightsController($user);
        $rightsController->makeClientAdmin();

        # Bearbeiter anlegen
        $operator = new Operator();
        $operator->name = $request->get("name");
        $operator->firstname = $request->get("firstname");
        $operator->email = $request->get("email");
        $operator->user_id = $user->id;
        $operator->client_id = $client->id;

        try {
            $operator->save();
        } catch (\Exception $e) {
            Log::error("Fehler beim Zuordnen eines neuen Administrators zu einem neuen Bearbeiter bei Mandant " . $tokenEntry->client_id .
                " (User" . $user->id);
            Log::error($e->getMessage());
            return redirect()->route('indexOutside')->withErrors(["Es ist ein Fehler aufgetreten."]);
        }

        # E-Mail-Adresse zur Bestätigung versenden
        $completeRegisterLink = URL::asset("/client/newClient/complete/" . $token);
        $confirmClientAdminMail = new ConfirmClientAdmin($user, $client->name, $completeRegisterLink);

        try {
            $job = new SendConfirmClientAdminEMail($user->email, $confirmClientAdminMail);
            dispatch($job)->onQueue('emails');
        } catch (\Exception $e) {
            Log::error("Fehler beim Versenden der E-Mail für das Registrieren eines neuen Mandantenadministrators");
            Log::error($e->getMessage());
            return redirect()->route('indexOutside')->withErrors(["Es ist ein Fehler aufgetreten."]);
        }

        # Token aktualisieren
        $tokenEntry->type = "completeData";
        try {
            $tokenEntry->save();
        } catch (\Exception $e) {
            Log::error("Fehler beim Aktualisieren des Tokens " . $tokenEntry->id);
            Log::error($e->getMessage());
            return redirect()->route('indexOutside')->withErrors(["Es ist ein Fehler aufgetreten."]);
        }

        # Auf Startseite weiterleiten
        $message = "An die E-Mail-Adresse " . $user->email . " wurde ein Link zur Bestätigung versandt. Danach kann mit der
        Vervollständigung der Daten von " . $client->name . " fortgefahren werden. Du musst dich dafür mit deinen Daten einloggen.";

        return redirect()->route('login')->with('status', $message);
    }


    # E-Mail-Adresse des Mandantenadministrators ist bestätigt
    # Formular zum Vervollständigen der Mandantenstammdaten anzeigen
    public function confirmClientAdmin($token) {
        $tokenEntry = ClientTokenController::checkTokenValidity("completeData", $token);

        if (!$tokenEntry)
            return redirect()->route('index')->withErrors(["Der Link zur Vervollständigung der Mandantendaten 
                                                                         ist nicht mehr gültig."]);

        # Adminrechte vergeben
        $loggedUser = User::find(Auth::user()->id);

        # Sicherhaltshalber prüfen, ob der eingeloggte User dem Mandanten auch zugeordnet ist
        if ($loggedUser->client->id != $tokenEntry->client_id)
            return redirect()->route('index')->withErrors(["Keine Berechtigung!"]);

        $rightsController = new RightsController($loggedUser);
        if (!$rightsController->makeClientAdmin())
            return redirect()->route('index')->withErrors(["Es ist ein Fehler aufgetreten."]);

        # Mandantendaten für Formularvorbelegung laden
        $client = Client::find($tokenEntry->client_id);
        if ($client == null) {
            Log::error("Kürzlich registrierter Mandant " . $tokenEntry->client_id . " nicht mehr gefunden!");
            return redirect()->route('index')->withErrors(["Es ist ein Fehler aufgetreten"]);
        }

        # Validierung erstellen
        $frontValidator = JsValidator::make($this->getUpdateValidationRules(), [],
            [], "#clientUpdate");

        # Formular für Stammdaten anzeigen
        return View::make('client.edit')->with(["client" => $client, "token" => $token,
            "freshRegister" => true, "validator" => $frontValidator]);
    }

    # Mandantenstammdaten vervollständigen
    public function updateClient(Request $request) {
        $loggedUser = User::find(Auth::user()->id);
        $client_id = $request->get("id");

        # Berechtigung prüfen
        if (!UserController::isClientAdmin($loggedUser))
            return redirect()->back()->withErrors(["Keine Berechtigung!"]);

        # Prüfen, ob eingeloggter User dem Mandanten zugeordnet ist
        if ($loggedUser->client->id != $client_id)
            return redirect()->route('index')->withErrors(["Keine Berechtigung!"]);

        # Mandantenobjekt laden
        $client = Client::find($client_id);

        if ($client == null) {
            Log::error("Kürzlich registrierter Mandant " . $client_id . " nicht mehr gefunden!");
            return redirect()->route('index')->withErrors(["Es ist ein Fehler aufgetreten"]);
        }

        # Formulardaten validieren
        $backValidator = Validator::make(request()->all(), $this->getUpdateValidationRules());
        if ($backValidator->fails())
            return redirect()->route('index')->withErrors($backValidator->errors());

        # Daten setzen
        $client->name = $request->get("name");
        $client->street = $request->get("street");
        $client->houseNumber = $request->get("houseNumber");
        $client->zipcode = $request->get("zipcode");
        $client->place = $request->get("place");
        $client->country = $request->get("country");
        $client->vatNo = $request->get("vatNo");

        # Daten speichern
        try {
            $client->save();
        } catch (\Exception $e) {
            Log::error("Fehler beim Speichern von Mandantenstammdaten von Mandant " . $client_id);
            Log::error($e->getMessage());
            return redirect()->route("index")->withErrors(["Es ist ein Fehler aufgetreten."]);
        }

        if ($request->get("freshRegister")) {
            # Token löschen
            try {
                $tokenEntry = ClientToken::where("client_id", $client_id)->where("type", "completeData");
                $tokenEntry->delete();
            } catch (\Exception $e) {
                # Wenn ein Token nicht gelöscht werden kann, ist das nicht weiter schlimm, beim nächsten Aufräumjob wird es
                # ohnehin gelöscht.
                Log::error("Fehler beim Löschen von Token.");
                Log::error($e->getMessage());
            }

            # Wenn Daten korrekt hinterlegt wurden, Mandanten aktivieren (nur bei Neuregistrierung)
            if (!$this->activateClient($client))
                return redirect()->route('index')->withErrors(["Es ist ein Fehler aufgetreten"]);

            $message = "Die Mandantendaten sind nun vollständig und die Registrierung ist abgeschlossen. Viel Spaß bei der Nutzung!";
            return redirect()->route('index')->with("message", $message);
        }
        else {
            $message = "Die Mandantenstammdaten wurden aktualisiert.";
            return redirect()->route('client.edit', ["client_id" => $client_id])->with("message", $message);
        }

    }

    public function activateClient(Client $client) {
        $client->firstContactMail = "";
        $client->activated = true;
        try {
            $client->save();
        } catch (\Exception $e) {
            Log::error("Fehler beim Aktivieren des Mandanten " . $client->id);
            Log::error($e->getMessage());
            return false;
        }
        return true;
    }


    public static function addUserToClient(Client $client, User $user) {
        return $client->user()->save($user);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Client $client
     * @return \Illuminate\Http\Response
     */
    public function edit($client_id = 0) {
        $user = User::find(Auth::user()->id);

        # Überprüfen, ob Mandanten-ID übergeben wurde: Dann darf nur Super-Admin bearbeiten!
        if ($client_id <> 0) {
            if (!UserController::isSuperAdmin($user)) {
                return redirect()->back()->withErrors(["Keine Berechtigung!"]);
            }
        }
        else {
            # Berechtigung prüfen, ob Nutzer Mandanten-Admin ist
            if (!UserController::isClientAdmin($user)) {
                return redirect()->back()->withErrors(["Keine Berechtigung!"]);
            }

            # Client-ID aus Benutzerobjekt holen
            $client_id = $user->client;
        }

        # Mandantendaten für Formularvorbelegung laden
        $client = Client::find($client_id);

        if ($client == null) {
            Log::error("Mandant " . $client_id . " nicht mehr gefunden!");
            return redirect()->route('index')->withErrors(["Es ist ein Fehler aufgetreten"]);
        }

        # Validierung erstellen
        $frontValidator = JsValidator::make($this->getUpdateValidationRules(), [],
            [], "#clientUpdate");

        $arguments = ["client" => $client,
            "token" => NULL,
            "freshRegister" => false,
            "validator" => $frontValidator];

        # Formular für Stammdaten anzeigen
        return View::make('client.edit')->with($arguments);
    }
}
