<?php

namespace App\Http\Controllers;

use App\NumberRanges;
use Illuminate\Http\Request;

use App\User;
use App\Client;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use JsValidator;

class NumberRangesController extends Controller {

    private $numberRangeValidationRules = [
        "current_number" => "required|numeric"
    ];

    /**
     * Display a listing of the resource.
     *
     */
    public function index() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("seeNumberRanges", $loggedUser))
            return redirect()->route('index');

        # Nummernkreise des Mandanten auslesen
        $numberRanges = $loggedUser->client->numberRanges;

        return View::make('number_ranges.list')->with(["numberRanges" => $numberRanges]);
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function edit($numberRangeId) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("editNumberRanges", $loggedUser))
            return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

        # Nummernkreis finden
        $numberRange = NumberRanges::find($numberRangeId);

        if ($numberRange == null) {
            return response()->json(['response' => 'error', 'message' => 'Nummernkreis nicht gefunden.']);
        }

        # Validierung
        $frontValidator = JsValidator::make($this->numberRangeValidationRules, [], [], "#editNumberRange");

        $arguments = ["numberRange" => $numberRange,
            "validator" => $frontValidator];

        # View rendern
        $editorHTML = View::make('number_ranges.edit')->with($arguments)->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }


    /**
     * Update the specified resource
     *
     */
    public function update(Request $request, $numberRangeId) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("editNumberRanges", $loggedUser))
            return redirect()->route('index');

        # Auftragstyp lesen
        $numberRange = NumberRanges::find($numberRangeId);

        if ($numberRange == null) {
            return redirect()->back()->withErrors("Nummernkreis nicht gefunden.");
        }

        # Speichern
        $trySave = $this->save($request, $loggedUser->client, $numberRange);
        if ($trySave instanceof NumberRanges)
            return redirect()->route('numberRanges.list')->with("message", "Nummernkreis wurde erfolgreich gespeichert.");
        # Wenn Speichern nicht erfolgreich, enthält $trySave eventuelle Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            return redirect()->back()->withErrors($trySave);
        }
    }


    /**
     * Save the specified resource in storage.
     *
     */
    public function save(Request $request, Client $client, NumberRanges $numberRange = null) {
        if (is_null($numberRange)) $numberRange = new NumberRanges();

        # Input validieren
        $backValidator = Validator::make($request->all(), $this->numberRangeValidationRules);

        if ($backValidator->fails()) {
            return $backValidator->errors();
        }

        # Neue Daten setzen
        $numberRange->client_id = $client->id;
        $numberRange->current_number = $request->get('current_number');

        # Speichern
        try {
            $numberRange->save();
        } catch (\Exception $e) {
            Log::error("Fehler beim Speichern der Daten für Nummernkreis " . $numberRange->id . ", Mandant " .
                $client->id . ": " . $e->getMessage());
            return ["Es ist ein Fehler aufgetreten."];
        }

        return $numberRange;
    }


}
