<?php

namespace App\Http\Controllers;

use App\OrderStatus;
use App\OrderTypes;
use Illuminate\Http\Request;

use App\User;
use App\Client;
use App\ProjectType;
use App\OperatingInstruction;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use JsValidator;

class OperatingInstructionsController extends Controller {

    private $operatingInstructionValidationRules = [
        'title' => 'required|string|max:255',
    ];

    /**
     * Display a listing of the resource.
     */
    public function index() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("seeOrderTypes", $loggedUser))
            return redirect()->route('index');

        # Liste aller OperatingInstructions(Arbeitsanweisungen) laden
        $operatingInstructionList = OperatingInstruction::all();

        return View::make('operating_instruction.list')->with(["operatingInstructionList" => $operatingInstructionList]);
    }

    /**
     * Show the form for creating a new resource.
     * Show the form for editing the specified resource.
     *
     */
    public function edit() {
        $loggedUser = User::find(Auth::user()->id);

        $operatingInstructionId = request()->input('operatingInstructionId');

        # Neue Arbeitsanweisung erstellen
        if ($operatingInstructionId == 0) {
            # Berechtigung prüfen
            if (!UserController::checkRight("createOrderTypes", $loggedUser))
                return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

            $operatingInstruction = new ProjectType();
        } # Arbeitsanweisung bearbeiten
        else {
            # Berechtigung prüfen
            if (!UserController::checkRight("editOrderTypes", $loggedUser))
                return response()->json(['response' => 'error', 'message' => ' Keine Berechtigung.']);

            # Arbeitsanweisung finden
            $operatingInstruction = OperatingInstruction::find($operatingInstructionId);
            if ($operatingInstruction == null)
                return response()->json(['response' => 'error', 'message' => 'Arbeitsanweisung nicht gefunden.']);

        }

        # Auftragsstatus auslesen (für Belegung Startstatus)
//        $orderStatusList = $loggedUser->client->order_status;

        # Validierung
        $frontValidator = JsValidator::make($this->operatingInstructionValidationRules, [], [], "#editOperatingInstruction");

        $arguments = ["operatingInstruction" => $operatingInstruction,
            "validator" => $frontValidator,
//            "orderStatusList" => $orderStatusList
        ];

        # View rendern
        $editorHTML = View::make('operating_instruction.edit')->with($arguments)->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }


    /**
     * Store a newly created resource
     *
     */
    public function store(Request $request) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("createOrderTypes", $loggedUser))
            return redirect()->route('index');

        # Speichern
        $trySave = $this->save($request, $loggedUser->client);
        if ($trySave instanceof OperatingInstruction)
            return redirect()->route('operatinginstructions.list')->with("message", "Arbeitsanweisung " . $trySave->title . " wurde erfolgreich angelegt.");
        # Wenn Speichern nicht erfolgreich, enthält $trySave eventuelle Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            return redirect()->back()->withErrors($trySave);
        }
    }


    /**
     * Update the specified resource
     *
     */
    public function update(Request $request, $operatingInstructionId) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("editOrderTypes", $loggedUser))
            return redirect()->route('index');

        # Arbeitsanweisung lesen
        $operatingInstruction = OperatingInstruction::find($operatingInstructionId);
        if ($operatingInstruction == null) {
            return redirect()->back()->withErrors("Arbeitsanweisung nicht gefunden.");
        }

        # Speichern
        $trySave = $this->save($request, $loggedUser->client, $operatingInstruction);
        if ($trySave instanceof OperatingInstruction)
            return redirect()->route('operatinginstructions.list')->with("message", "Arbeitsanweisung " . $operatingInstruction->title . " wurde erfolgreich gespeichert.");
        # Wenn Speichern nicht erfolgreich, enthält $trySave eventuelle Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            return redirect()->back()->withErrors($trySave);
        }
    }


    /**
     * Save the specified resource in storage.
     *
     */
    public function save(Request $request, Client $client, OperatingInstruction $operatingInstruction = null) {
        if (is_null($operatingInstruction)) $operatingInstruction = new OperatingInstruction();

        # Input validieren
        $backValidator = Validator::make($request->all(), $this->operatingInstructionValidationRules);

        if ($backValidator->fails()) {
            return $backValidator->errors();
        }

        # Neue Daten setzen
//        $orderType->client_id = $client->id;
        $operatingInstruction->title = $request->get('title');
//        $orderType->order_statuses_id = $request->get('order_statuses_id');

        # Speichern
        try {
            $operatingInstruction->save();
        } catch (\Exception $e) {
            Log::error("Fehler beim Speichern der Daten für Arbeitsanweisung " . $operatingInstruction->title . ", (ID: " . $operatingInstruction->id . " ), Mandant " .
                $client->id . ": " . $e->getMessage());
            return "Es ist ein Fehler aufgetreten.";
        }

        return $operatingInstruction;
    }


    public function showDeleteConfirm() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("deleteOrderTypes", $loggedUser))
            return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

        $operatingInstructionId = request()->input('operatingInstructionId');

        # Arbeitsanweisung lesen
        $operatingInstruction = OperatingInstruction::find($operatingInstructionId);
        if ($operatingInstruction == null)
            return response()->json(['response' => 'error', 'message' => 'Arbeitsanweisung nicht gefunden.']);

        $arguments = ["confItem" => "operationInstructions",
            "confItemLabel" => "Arbeitsanweisung",
            "confItemLabelArticle" => "die",
            "additional_info" => "",
            "item" => $operatingInstruction];

        # View rendern
        $editorHTML = View::make('project_conf.delete')->with($arguments)->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }

    /**
     * Remove the specified resource from storage.
     *
     */
    public function destroy() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::isClientAdmin($loggedUser) && !UserController::checkRight("deleteOrderTypes", $loggedUser))
            return redirect()->route('index');

        $operatingInstructionId = request()->input('operationInstructionsId');
        $confirmDelete = request()->input('confirmDelete');

        # Arbeitsanweisung lesen
        $operatingInstruction = OperatingInstruction::find($operatingInstructionId);
        if ($operatingInstruction == null)
            return redirect()->back()->withErrors("Arbeitsanweisung nicht gefunden.");

        # Prüfen, ob Checkbox bestätigt ist
        if (!is_null($confirmDelete)) {
            # Arbeitsanweisung löschen - wenn möglich!
            try {
                # Versuchen, zu löschen
                $operatingInstruction->delete();
            } catch (\Exception $e) {
                # Es kann sein, dass noch Relations zu Dokumenten bestehen
                Log::error("Fehler beim Löschen der Arbeitsanweisung " . $operatingInstruction->id . ", Mandant " .
                    $loggedUser->client->id . ": " . $e->getMessage());

                return redirect()->back()->withErrors("Arbeitsanweisung konnte nicht gelöscht werden. Es bestehen noch 
            Verbindungen zu Dokumenten oder anderen Tabellen.");
            }
        }
        else {
            return redirect()->back()->with("message", "Die Arbeitsanweisung wurde nicht gelöscht, da das Löschen bestätigt werden muss!");
        }

        Log::info("Arbeitsanweisung " . $operatingInstruction->id . ", Mandant " . $loggedUser->client->id . " wurde gelöscht. 
                            (User " . $loggedUser->id . ")");
        return redirect()->route('operatinginstructions.list')->with("message", "Arbeitsanweisung wurde erfolgreich gelöscht.");
    }
}
