<?php

namespace App\Http\Controllers;

use App\OrderPaymentStatus;
use Illuminate\Http\Request;

use App\User;
use App\Client;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use JsValidator;

class OrderPaymentStatusController extends Controller {

    private $orderPaymentStatusValidationRules = [
        'label' => 'required|string|max:255',
    ];

    /**
     * Display a listing of the resource.
     */
    public function index() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("seeOrderPaymentStatus", $loggedUser))
            return redirect()->route('index');

        # Bezahlstatus des Mandanten auslesen
        $orderPaymentStatus = $loggedUser->client->order_payment_status;

        return View::make('order_conf.order_payment_status.list')->with(["orderPaymentStatus" => $orderPaymentStatus]);
    }


    /**
     * Show the form for creating a new resource.
     * Show the form for editing the specified resource.
     *
     */
    public function edit() {
        $loggedUser = User::find(Auth::user()->id);

        $orderPaymentStatusId = request()->input('paymentStatusId');

        # Neuen Bezahlstatus erstellen
        if ($orderPaymentStatusId == 0) {
            # Berechtigung prüfen
            if (!UserController::checkRight("createOrderPaymentStatus", $loggedUser))
                return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

            $orderPaymentStatus = new OrderPaymentStatus();
        }
        # Bezahlstatus bearbeiten
        else {
            # Berechtigung prüfen
            if (!UserController::checkRight("editOrderPaymentStatus", $loggedUser))
                return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

            # Bezahlstatus finden
            $orderPaymentStatus = OrderPaymentStatus::find($orderPaymentStatusId);
            if ($orderPaymentStatus == null) {
                return response()->json(['response' => 'error', 'message' => 'Bezahlstatus nicht gefunden.']);
            }
        }

        # Validierung
        $frontValidator = JsValidator::make($this->orderPaymentStatusValidationRules, [], [], "#editOrderPaymentStatus");

        # View rendern
        $editorHTML = View::make('order_conf.order_payment_status.edit')->with(["orderPaymentStatus" => $orderPaymentStatus, "validator" => $frontValidator])->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }


    /**
     * Store a newly created resource
     *
     */
    public function store(Request $request) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung nochmal prüfen
        if (!UserController::checkRight("createOrderPaymentStatus", $loggedUser))
            return redirect()->route('index');

        # Speichern
        $trySave = $this->save($request, $loggedUser->client);
        if ($trySave instanceof OrderPaymentStatus)
            return redirect()->route('orderPaymentStatus.list')->with("message", "Bezahlstatus " . $trySave->label . " wurde erfolgreich angelegt.");
        # Wenn Speichern nicht erfolgreich, enthält $trySave eventuelle Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            return redirect()->back()->withErrors($trySave);
        }
    }


    /**
     * Update the specified resource
     *
     */
    public function update(Request $request, $orderPaymentStatusId) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("editOrderPaymentStatus", $loggedUser))
            return redirect()->route('index');

        # Bezahlstatus lesen
        $orderPaymentStatus = OrderPaymentStatus::find($orderPaymentStatusId);
        if ($orderPaymentStatus == null) {
            return redirect()->back()->withErrors("Bezahlstatus nicht gefunden.");
        }

        # Speichern
        $trySave = $this->save($request, $loggedUser->client, $orderPaymentStatus);
        if ($trySave instanceof OrderPaymentStatus)
            return redirect()->route('orderPaymentStatus.list')->with("message", "Bezahlstatus " . $orderPaymentStatus->label . " wurde erfolgreich gespeichert.");
        # Wenn Speichern nicht erfolgreich, enthält $trySave eventuelle Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            return redirect()->back()->withErrors($trySave);
        }
    }


    /**
     * Save the specified resource in storage.
     *
     */
    public function save(Request $request, Client $client, OrderPaymentStatus $orderPaymentStatus = null) {
        if (is_null($orderPaymentStatus)) $orderPaymentStatus = new OrderPaymentStatus();

        # Input validieren
        $backValidator = Validator::make($request->all(), $this->orderPaymentStatusValidationRules);

        if ($backValidator->fails()) {
            return $backValidator->errors();
        }

        # Neue Daten setzen
        $orderPaymentStatus->client_id = $client->id;
        $orderPaymentStatus->label = $request->get('label');

        # Speichern
        try {
            $orderPaymentStatus->save();
        } catch (\Exception $e) {
            Log::error("Fehler beim Speichern der Daten für Bezahlstatus " . $orderPaymentStatus->label . ", (ID: " . $orderPaymentStatus->id . " ), Mandant " .
                $client->id . ": " . $e->getMessage());
            return "Es ist ein Fehler aufgetreten.";
        }

        return $orderPaymentStatus;
    }

    public function showDeleteConfirm() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("deleteOrderPaymentStatus", $loggedUser))
            return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

        $orderPaymentStatusId = request()->input('paymentStatusId');

        # Bezahlstatus lesen
        $orderPaymentStatus = OrderPaymentStatus::find($orderPaymentStatusId);
        if ($orderPaymentStatus == null)
            return response()->json(['response' => 'error', 'message' => 'Bezahlstatus nicht gefunden.']);

        $arguments = ["confItem" => "orderPaymentStatus",
            "confItemLabel" => "Bezahlstatus",
            "confItemLabelArticle" => "den",
            "item" => $orderPaymentStatus];

        # View rendern
        $editorHTML = View::make('order_conf.delete')->with($arguments)->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }


    /**
     * Remove the specified resource from storage.
     *
     */
    public function destroy() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("deleteOrderPaymentStatus", $loggedUser))
            return redirect()->route('index');

        $orderPaymentStatusId = request()->input('orderPaymentStatusId');
        $confirmDelete = request()->input('confirmDelete');

        # Bezahlstatus lesen
        $orderPaymentStatus = OrderPaymentStatus::find($orderPaymentStatusId);
        if ($orderPaymentStatus == null)
            return redirect()->back()->withErrors("Bezahlstatus nicht gefunden.");


        # Prüfen, ob Checkbox bestätigt ist
        if (!is_null($confirmDelete)) {
            # Bezahlstatus löschen - wenn möglich!
            try {
                # Versuchen, zu löschen
                $orderPaymentStatus->delete();
            } catch (\Exception $e) {
                # Es kann sein, dass noch Relations zu Aufträgen (orders) bestehen
                Log::error("Fehler beim Löschen des Bezahlstatus " . $orderPaymentStatus->id . ", Mandant " .
                    $loggedUser->client->id . ": " . $e->getMessage());

                return redirect()->back()->withErrors("Bezahlstatus konnte nicht gelöscht werden. Es bestehen noch 
            Verbindungen zu Aufträgen.");
            }
        }
        else {
            return redirect()->back()->with("message", "Der Bezahlstatus wurde nicht gelöscht, da das Löschen bestätigt werden muss!");
        }

        Log::info("Bezahlstatus " . $orderPaymentStatus->id . ", Mandant " . $loggedUser->client->id . " wurde gelöscht. 
                            (User " . $loggedUser->id . ")");
        return redirect()->route('orderPaymentStatus.list')->with("message", "Bezahlstatus wurde erfolgreich gelöscht.");
    }
}
