<?php

namespace App\Http\Controllers;

use App\OrderStatus;
use App\OrderTypes;
use Illuminate\Http\Request;

use App\User;
use App\Client;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use JsValidator;

class OrderTypesController extends Controller {

    private $orderTypeValidationRules = [
        'label' => 'required|string|max:255',
    ];

    /**
     * Display a listing of the resource.
     */
    public function index() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("seeOrderTypes", $loggedUser))
            return redirect()->route('index');

        # Auftragstypen des Mandanten auslesen
        $orderTypes = $loggedUser->client->order_types;

        return View::make('order_conf.order_types.list')->with(["orderTypes" => $orderTypes]);
    }

    /**
     * Show the form for creating a new resource.
     * Show the form for editing the specified resource.
     *
     */
    public function edit() {
        $loggedUser = User::find(Auth::user()->id);

        $orderTypeId = request()->input('orderTypeId');

        # Neuen Auftragstyp erstellen
        if ($orderTypeId == 0) {
            # Berechtigung prüfen
            if (!UserController::checkRight("createOrderTypes", $loggedUser))
                return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

            $orderType = new OrderTypes();
        } # Auftragstyp bearbeiten
        else {
            # Berechtigung prüfen
            if (!UserController::checkRight("editOrderTypes", $loggedUser))
                return response()->json(['response' => 'error', 'message' => ' Keine Berechtigung.']);

            # Auftragstyp finden
            $orderType = OrderTypes::find($orderTypeId);
            if ($orderType == null)
                return response()->json(['response' => 'error', 'message' => 'Auftragstyp nicht gefunden.']);

        }

        # Auftragsstatus auslesen (für Belegung Startstatus)
        $orderStatusList = $loggedUser->client->order_status;

        # Validierung
        $frontValidator = JsValidator::make($this->orderTypeValidationRules, [], [], "#editOrderType");

        $arguments = ["orderType" => $orderType,
            "validator" => $frontValidator,
            "orderStatusList" => $orderStatusList];

        # View rendern
        $editorHTML = View::make('order_conf.order_types.edit')->with($arguments)->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }


    /**
     * Store a newly created resource
     *
     */
    public function store(Request $request) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("createOrderTypes", $loggedUser))
            return redirect()->route('index');

        # Speichern
        $trySave = $this->save($request, $loggedUser->client);
        if ($trySave instanceof OrderTypes)
            return redirect()->route('orderTypes.list')->with("message", "Auftragstyp " . $trySave->label . " wurde erfolgreich angelegt.");
        # Wenn Speichern nicht erfolgreich, enthält $trySave eventuelle Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            return redirect()->back()->withErrors($trySave);
        }
    }


    /**
     * Update the specified resource
     *
     */
    public function update(Request $request, $orderTypeId) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("editOrderTypes", $loggedUser))
            return redirect()->route('index');

        # Auftragstyp lesen
        $orderType = OrderTypes::find($orderTypeId);
        if ($orderType == null) {
            return redirect()->back()->withErrors("Auftragstyp nicht gefunden.");
        }

        # Speichern
        $trySave = $this->save($request, $loggedUser->client, $orderType);
        if ($trySave instanceof OrderTypes)
            return redirect()->route('orderTypes.list')->with("message", "Auftragstyp " . $orderType->label . " wurde erfolgreich gespeichert.");
        # Wenn Speichern nicht erfolgreich, enthält $trySave eventuelle Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            return redirect()->back()->withErrors($trySave);
        }
    }


    /**
     * Save the specified resource in storage.
     *
     */
    public function save(Request $request, Client $client, OrderTypes $orderType = null) {
        if (is_null($orderType)) $orderType = new OrderTypes();

        # Input validieren
        $backValidator = Validator::make($request->all(), $this->orderTypeValidationRules);

        if ($backValidator->fails()) {
            return $backValidator->errors();
        }

        # Neue Daten setzen
        $orderType->client_id = $client->id;
        $orderType->label = $request->get('label');
        $orderType->order_statuses_id = $request->get('order_statuses_id');

        # Speichern
        try {
            $orderType->save();
        } catch (\Exception $e) {
            Log::error("Fehler beim Speichern der Daten für Auftragstyp " . $orderType->label . ", (ID: " . $orderType->id . " ), Mandant " .
                $client->id . ": " . $e->getMessage());
            return "Es ist ein Fehler aufgetreten.";
        }

        return $orderType;
    }


    public function showDeleteConfirm() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("deleteOrderTypes", $loggedUser))
            return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

        $orderTypeId = request()->input('orderTypeId');

        # Auftragstyp lesen
        $orderType = OrderTypes::find($orderTypeId);
        if ($orderType == null)
            return response()->json(['response' => 'error', 'message' => 'Auftragstyp nicht gefunden.']);

        $arguments = ["confItem" => "orderTypes",
            "confItemLabel" => "Auftragstyp",
            "confItemLabelArticle" => "den",
            "item" => $orderType];

        # View rendern
        $editorHTML = View::make('order_conf.delete')->with($arguments)->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }

    /**
     * Remove the specified resource from storage.
     *
     */
    public function destroy() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::isClientAdmin($loggedUser) && !UserController::checkRight("deleteOrderTypes", $loggedUser))
            return redirect()->route('index');

        $orderTypeId = request()->input('orderTypesId');
        $confirmDelete = request()->input('confirmDelete');

        # Auftragstyp lesen
        $orderType = OrderTypes::find($orderTypeId);
        if ($orderType == null)
            return redirect()->back()->withErrors("Auftragstyp nicht gefunden.");

        # Prüfen, ob Checkbox bestätigt ist
        if (!is_null($confirmDelete)) {
            # Auftragstyp löschen - wenn möglich!
            try {
                # Versuchen, zu löschen
                $orderType->delete();
            } catch (\Exception $e) {
                # Es kann sein, dass noch Relations zu Auftragskennzeichen (order_tags) oder Aufträgen (orders) bestehen
                Log::error("Fehler beim Löschen des Auftragstyps " . $orderType->id . ", Mandant " .
                    $loggedUser->client->id . ": " . $e->getMessage());

                return redirect()->back()->withErrors("Auftragstyp konnte nicht gelöscht werden. Es bestehen noch 
            Verbindungen zu Aufträgen oder Auftragskennzeichen.");
            }
        }
        else {
            return redirect()->back()->with("message", "Der Auftragstyp wurde nicht gelöscht, da das Löschen bestätigt werden muss!");
        }

        Log::info("Auftragstyp " . $orderType->id . ", Mandant " . $loggedUser->client->id . " wurde gelöscht. 
                            (User " . $loggedUser->id . ")");
        return redirect()->route('orderTypes.list')->with("message", "Auftragstyp wurde erfolgreich gelöscht.");
    }
}
