<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use App\User;
use App\Client;
use App\Product;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use JsValidator;

class ProductsController extends Controller {
    private $validationRules = [
        "product_number" => "required|string|max:50",
        "name" => "required|string|max:70",
        "price_1" => "required|numeric",
        "price_2" => "required|numeric",
        "price_3" => "required|numeric",
        "price_4" => "required|numeric",
    ];

    /**
     * Display a listing of the resource.
     *
     */
    public function index() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("seeProducts", $loggedUser))
            return redirect()->route('index');

        # Mögliche Produkte auslesen
        $products = Product::where("client_id", $loggedUser->client->id)->get();

        return View::make('order_conf.products.list')->with(["products" => $products]);

    }

    /**
     * Show the form for creating a new resource.
     * Show the form for editing the specified resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function edit($productId) {
        $loggedUser = User::find(Auth::user()->id);

        # Neuen Artikel erstellen
        if ($productId == 0) {
            # Berechtigung prüfen
            if (!UserController::checkRight("createProducts", $loggedUser))
                return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

            $product = new Product();
        }
        # Artikel bearbeiten
        else {
            # Berechtigung prüfen
            if (!UserController::checkRight("editProducts", $loggedUser))
                return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

            # Artikel finden
            $product = Product::find($productId);
            if ($product == null)
                return response()->json(['response' => 'error', 'message' => 'Artikel nicht gefunden.']);
        }

        # Validierung
        $frontValidator = JsValidator::make($this->validationRules, [], [], "#editProduct");

        $arguments = ["product" => $product,
            "validator" => $frontValidator];

        # View rendern
        $editorHTML = View::make('order_conf.products.edit')->with($arguments)->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }

    /**
     * Store a newly created resource
     *
     */
    public function store(Request $request) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("createProducts", $loggedUser))
            return redirect()->route('index');

        # Speichern
        $trySave = $this->save($request, $loggedUser->client);
        if ($trySave instanceof Product)
            return redirect()->route('products.list')->with("message", "Artikel wurde erfolgreich angelegt.");
        # Wenn Speichern nicht erfolgreich, enthält $trySave eventuelle Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            return redirect()->back()->withErrors($trySave);
        }
    }

    /**
     * Update the specified resource
     *
     */
    public function update(Request $request, $productId) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("editProducts", $loggedUser))
            return redirect()->route('index');

        # Artikel lesen
        $product = Product::find($productId);

        if ($product == null)
            return redirect()->back()->withErrors("Artikel nicht gefunden.");

        # Speichern
        $trySave = $this->save($request, $loggedUser->client, $product);
        if ($trySave instanceof Product)
            return redirect()->route('products.list')->with("message", "Artikel wurde erfolgreich gespeichert.");
        # Wenn Speichern nicht erfolgreich, enthält $trySave eventuelle Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            return redirect()->back()->withErrors($trySave);
        }
    }

    /**
     * Save the specified resource in storage.
     *
     */
    public function save(Request $request, Client $client, Product $product = null) {
        if (is_null($product)) $product = new Product();

        # Input validieren
        $backValidator = Validator::make($request->all(), $this->validationRules);

        if ($backValidator->fails())
            return $backValidator->errors();


        # Neue Daten setzen
        $product->client_id = $client->id;
        $product->product_number = $request->get('product_number');
        $product->name = $request->get('name');

        $product->price_1 = $request->get('price_1');
        $product->price_2 = $request->get('price_2');
        $product->price_3 = $request->get('price_3');
        $product->price_4 = $request->get('price_4');

        # Speichern
        try {
            $product->save();
        } catch (\Exception $e) {
            Log::error("Fehler beim Speichern der Daten für Artikel " . $product->name . ", (ID: " . $product->id . " ), Mandant " .
                $client->id . ": " . $e->getMessage());
            return ["Es ist ein Fehler aufgetreten."];
        }

        return $product;
    }

    public function showDeleteConfirm() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("deleteProducts", $loggedUser))
            return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

        $productId = request()->input('productId');

        # Produkt lesen
        $product = Product::find($productId);
        if ($product == null)
            return response()->json(['response' => 'error', 'message' => 'Artikel nicht gefunden.']);

        $arguments = ["confItem" => "products",
            "confItemLabel" => "Artikel",
            "confItemLabelArticle" => "den",
            "item" => $product];

        # View rendern
        $editorHTML = View::make('order_conf.delete')->with($arguments)->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }


    /**
     * Remove the specified resource from storage.
     *
     */
    public function destroy() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("deleteProducts", $loggedUser))
            return redirect()->route('index');

        $productId = request()->input('productsId');
        $confirmDelete = request()->input('confirmDelete');

        # Artikel lesen
        $product = Product::find($productId);
        if ($product == null)
            return redirect()->back()->withErrors("Artikel nicht gefunden.");


        # Prüfen, ob Checkbox bestätigt ist
        if (!is_null($confirmDelete)) {
            # Artikel löschen - wenn möglich!
            try {
                # Versuchen, zu löschen
                $product->delete();
            } catch (\Exception $e) {
                Log::error("Fehler beim Löschen des Artikels " . $product->id . ", Mandant " .
                    $loggedUser->client->id . ": " . $e->getMessage());

                return redirect()->back()->withErrors("Artikel konnte nicht gelöscht werden.");
            }
        }
        else {
            return redirect()->back()->with("message", "Der Artikel wurde nicht gelöscht, da das Löschen bestätigt werden muss!");
        }

        Log::info("Artikel " . $product->id . ", Mandant " . $loggedUser->client->id . " wurde gelöscht.
                                (User " . $loggedUser->id . ")");
        return redirect()->route('products.list')->with("message", "Artikel wurde erfolgreich gelöscht.");
    }

}
