<?php


namespace App\Http\Controllers;


use App\Jobs\ProjectInitializationJob;
use App\Operator;
use App\OrderAddresses;
use App\OrderCreditNote;
use App\OrderDocuments;
use App\OrderLogs;
use App\OrderPaymentStatus;
use App\OrderStatus;
use App\OrderTags;
use App\OrderTypes;
use App\Repositories\DocumentBaseRepository;
use App\Repositories\PowerPointDocumentRepository;
use App\Repositories\ProjectDocumentRepository;
use App\Repositories\ProjectTypeDocumentRepository;
use App\Variable;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use JsValidator;
use App\Customer;
use App\Orders;
use App\User;
use App\Project;
use App\ProjectType;
use App\Article;
use App\ArticleVariable;

use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\View;

class ProjectsController extends Controller{



    private $orderCreateSensibleValidationRules = [
        'customer_id' => 'required|min:1',
        'project_type_id' => 'required|min:1',
//        'order_value' => 'required|regex:/^\d*(\,\d{1,2})?$/',
//        'end_date' => 'required|date_format:"Y-m-d"',
//        'export_date' => 'required|date_format:"Y-m-d"',
    ];



    /**
     * Display a listing of the resource.
     *
     */
    public function index($customerId = NULL) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("seeOrdersList", $loggedUser))
            return redirect()->route('index');

        $parameters = self::getParametersForOrdersList($loggedUser);

        # Ggf. nach Auftraggeber vorsortieren
        if ($customerId <> NULL) {
            $customer = Customer::find($customerId);
            if ($customer <> NULL) {
                $parameters["preFilters"]["customer"] = $customerId;
                $parameters["preFilteredByCustomer"] = true;
                $parameters["preFilteredCustomer"] = $customer->company;
            }
        }

        # Darf nur Aufträge sehen, bei denen verantwortlich
        if (UserController::checkRight("seeOnlyOwnOrders", $loggedUser))
            $parameters["preFilters"]["responsible"] = $loggedUser->operator->id;


        return View::make('project.list')->with($parameters);
    }

    public static function getParametersForOrdersList($user) {
        # Für Filterung alle Auftragstypen etc. laden
        $parameters = [
            "orderStatusList" => $user->client->order_status,
            "orderPaymentStatusList" => $user->client->order_payment_status,
            "orderTypesList" => $user->client->order_types,
            "orderTagsList" => $user->client->order_tags];

        $parameters = array_merge(UserController::convertOrderRightsToJs($user), $parameters);

        return $parameters;
    }



    /* Auftragssuche via AJAX */
    public function asyncSearch() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("seeOrdersList", $loggedUser))
            return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

        $noResult = false;
        $now = microtime(true);


        $limit = request()->get("limit");
        $offset = request()->get("offset");
        $jsonSearch = request()->get("searchParameters");
        $jsonSort = request()->get("sortParameters");

        $result = array();

        $decodedSearchJson = json_decode($jsonSearch);
        $decodedSortJson = json_decode($jsonSort);

        if ($decodedSearchJson == null)
            return response()->json(['response' => 'error']);

        if ($decodedSortJson == null)
            return response()->json(['response' => 'error']);


        # Für leichtere Verarbeitung in Array casten
        $searchParameters = (array)$decodedSearchJson;
        $sortParameters = (array)$decodedSortJson;

        $result = Project::with(['customer', 'project_type', 'project_operator_responsible'])->orderByDesc("project_nr")->get();

        /*
         * FILTERUNG
         */
        $mode = "ordersList";
        $orderCreditNotesJoin = false;

        # Controlling: Monat
//        if (array_key_exists("month", $searchParameters) && array_key_exists("year", $searchParameters)) {
//            $firstDay = Carbon::createMidnightDate($searchParameters["year"], $searchParameters["month"], 1);
//            $searchParameters["dateFrom"] = $firstDay->toDateString();
//
//            $lastDay = $firstDay->endOfMonth();
//            $searchParameters["dateTo"] = $lastDay->toDateString();
//
//            $mode = "controllingList";
//        }

        # Auftragsnummer - wenn diese eingegeben wird, kommt genau ein Auftrag zurück, also den Rest ignorieren!
        if ($mode == "ordersList" &&
            (array_key_exists("ordernumber", $searchParameters) && $searchParameters["ordernumber"] <> "")) {
            $query = DB::table('orders')->where("order_number", $searchParameters["ordernumber"]);
            $result = $query->get();
        }

        # Wichtigste Daten für Listenvorblendung zusammensuchen
        $returnOrders = array();

        $offsetCounter = 0;
        $limitCounter = 0;

        $totalResult = 0;
        $totalSum = 0;
        $totalApartmentUnits = 0;

        foreach ($result as $item) {
            $totalResult++;

            if ($limit != -1 && $offsetCounter < $offset) {
                $offsetCounter++;
                continue;
            }


            if ($limit != -1 && $limitCounter >= $limit) continue;


            # Verantwortliche Bearbeiter
            $projectOperatorResponsibleArray = array();
            if (count($item->project_operator_responsible) > 0) {
                foreach ($item->project_operator_responsible as $projectOperatorResponsible) {
                    $projectOperatorResponsibleArray [$projectOperatorResponsible->id] = $projectOperatorResponsible->name;
                }
                $item->projectOperatorResponsibleList = json_encode($projectOperatorResponsibleArray );
            }
            else
                $item->projectOperatorResponsibleList = "{}";


            $returnOrders[] = $item;
            $limitCounter++;
        }

        # Auftragssumme ggf. runden und Dezimalstellen kürzen
        $totalSum = round($totalSum, 2);

        if (count($returnOrders) <= 0) $noResult = true;

        return response()->json(['response' => 'success', 'noResult' => $noResult, 'returnOrders' => json_encode($returnOrders),
            'totalResult' => $totalResult, 'totalSum' => $totalSum, 'totalApartmentUnits' => $totalApartmentUnits]);
    }

    /**
     * Show the form for creating a new resource.
     *
     */
    public function create() {
        $loggedUser = User::find(Auth::user()->id);

//        $testpp = new PowerPointDocumentRepository(storage_path("app/public/uploads" . DocumentBaseRepository::BASE_PATH . "test.pptx"));
//        $testpp->modifyDocument();

        # Berechtigung prüfen
        if (!UserController::checkRight("createOrders", $loggedUser))
            return redirect()->route('index');

        # Zugehörigen Mandanten bestimmen
        $client = $loggedUser->client;

        # Auftraggeberliste laden
//        $customerList = $client->customer;
        $customerList = Customer::all();

        # Auftragstypen laden
        $orderTypesList = $client->order_types;

        # Projekttypen laden
        $projectTypeList = ProjectType::all();

        # Auftragskennzeichen laden
        $orderTagsList = $client->order_tags;

        # Auftragsstatus laden
        $orderStatusList = $client->order_status;

        # Bezahlstatus laden
        $orderPaymentStatusList = $client->order_payment_status;

        # Bearbeiter laden
//        $operators = $client->operatorsWithUser;
        $operators = Operator::all();

        # Validierung
        $frontValidator = JsValidator::make($this->orderCreateSensibleValidationRules, [], [], "#createOrder");

        $arguments = ["customerList" => $customerList,
            "orderTypesList" => $orderTypesList,
            "orderTagsList" => $orderTagsList,
            "orderStatusList" => $orderStatusList,
            "orderPaymentStatusList" => $orderPaymentStatusList,
            "operatorsList" => $operators,
            "validator" => $frontValidator,
            "projectTypeList" => $projectTypeList
        ];

        return View::make('project.create')->with($arguments);
    }

    public function checkStatus($projectId)
    {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("editOrders", $loggedUser))
            return redirect()->route('index');


        # Auftragsdaten lesen
        $project = Project::find($projectId);
        if ($project == null)
            return "{}";

        return $project->toJson();
    }

    public function createArticle($projectId){
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("editOrders", $loggedUser))
            return redirect()->route('index');


        # Auftragsdaten lesen
        $project = Project::find($projectId);
        if ($project == null)
            return redirect()->route("projects.list")->withErrors("Projekt nicht gefunden.");

//        $projectType = $project->project_type;
//        $documentTemplateVariableList = $projectType->document_templates->pluck('variables');

        $variables = Variable::all();

        $parameters = ["projectToEdit" => $project,
              "variables" => $variables
//            "customerList" => $customerList,
//            "orderTypesList" => $projectTypeList,
//            "orderTagsList" => $orderTagsList,
//            "orderStatusList" => $orderStatusList,
//            "orderPaymentStatusList" => $orderPaymentStatusList,
//            "orderAddresses" => $orderAddresses,
//            "orderDocuments" => $orderDocuments,
//            "orderLogs" => $orderLogs,
//            "orderAppointments" => $orderAppointments,
//            "orderMailLogs" => $orderMailLogs,
//            "orderFollowUps" => $orderFollowUps,
//            "responsibleOperatorsList" => $responsibleOperators,
//            "operatorsList" => $operators,
//            "orderCreditNotes" => $orderCreditNotes,
//            "validator" => $frontValidator,
//            "orderInvoices" => $orderInvoices
        ];

        return View::make('project.createArticle')->with($parameters);
    }


    /**
     * Store a newly created resource in storage.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("createOrders", $loggedUser))
            return redirect()->route('index');

        # Input validieren, soweit möglich
        $backValidator = Validator::make($request->all(), $this->orderCreateSensibleValidationRules);

        if ($backValidator->fails()) {
            Log::error("Fehler beim Anlegen eines neuen Auftrags: Validierungsfehler");
            return redirect()->back()->withErrors($backValidator->errors());
        }

        $newProject = new Project();

        # Daten setzen
        $newProject->customer_id = $request->get("customer_id");
        $newProject->project_nr = "";
        $newProject->project_type_id = $request->get("project_type_id");
        $newProject->status = Project::$STATUS_DRAFT;


        # Projekt zwischenspeichern, um ID zu erhalten
        try {
            $newProject->save();
        } catch (\Exception $e) {
            Log::error("Fehler beim Anlegen eines neuen Projekts: " . $e->getMessage());
            return redirect()->back()->withErrors("Es ist ein Fehler aufgetreten");
        }

        $newProject->project_nr = "PR".str_pad($newProject->id, 10, "0", STR_PAD_LEFT);

        try {
            $newProject->save();
        } catch (\Exception $e) {
            Log::error("Fehler beim Anlegen eines neuen Projekts: " . $e->getMessage());
            return redirect()->back()->withErrors("Es ist ein Fehler aufgetreten");
        }


        # Diverse Komponenten zum Auftrag hinzufügen

        # Verantwortliche Bearbeiter
        $responsibleOperators = $request->get("responsibleOperators");

        $responsibleOperatorsError = false;
        $operatorsToAdd = array();
        if ($responsibleOperators <> NULL && count($responsibleOperators) > 0) {
            foreach ($responsibleOperators as $operator => $status) {
                if ($status != "on") continue;

                $addOperator = Operator::find($operator);
                if ($addOperator == null) continue;

                $operatorsToAdd[] = $addOperator->id;
            }

            try {
                $newProject->project_operator_responsible()->attach($operatorsToAdd);
            } catch (\Exception $e) {
                Log::error("Fehler beim Hinzufügen von verantwortlichen Bearbeitern: " . $e->getMessage());
                $responsibleOperatorsError = true;
            }
        }



        # Log schreiben
        OrderLogsController::writeLog(1, $newProject->id, $loggedUser->operator->id);

        # Erfolgsmeldung zurückgeben, ggf. mit Hinweisen
        $message = "Das Projekt " . $newProject->project_nr . " wurde erfolgreich angelegt. ";
//        if ($orderTagsError) $message .= "Es gab einen Fehler beim Anlegen der Auftragskennzeichen.";
//        if ($orderAddressError) $message .= "Es gab einen Fehler beim Anlegen der Auftragsadressen.";
//        if ($orderFilesError) $message .= "Es gab einen Fehler beim Zuordnen der Auftragsdokumente.";
//        if ($orderAppointmentError) $message .= "Es gab einen Fehler beim Zuweisen von einem Auftragstermin.";
//        if ($orderValueError) $message .= "Es gab einen Fehler beim  Hinterlegen des Auftragswerts.";
//        if ($orderEndDateError) $message .= "Es gab einen Fehler beim  Hinterlegen des Endtermins.";
//        if ($orderExportDateError) $message .= "Es gab einen Fehler beim  Hinterlegen des Exporttermins.";
        if ($responsibleOperatorsError) $message .= "Es gab einen Fehler beim  Hinterlegen der verantwortlichen Bearbeiter.";

        if ($responsibleOperatorsError)
            return redirect()->route('projectArticles.create')->with('okayWithErrors', $message);
        else
            return redirect()->route('projectArticles.create', $newProject->id)->with('message', $message);
    }

    public function storeArticles(Request $request)
    {
        $projectId = $request->get("projectToEdit");
        $articleVariables = $request->get("ordertagsValues");
        $tags = $request->get("ordertags");
        $articleNumbers = $articleVariables[0];

        $project = Project::find($projectId);
        if ($project == null)
            return redirect()->route("projects.list")->withErrors("Projekt nicht gefunden.");


        $project->status = Project::$STATUS_PENDING;


        # Projekt zwischenspeichern, um ID zu erhalten
        try {
            $project->save();
        } catch (\Exception $e) {
            Log::error("Fehler beim Anlegen eines neuen Projekts: " . $e->getMessage());
            return redirect()->back()->withErrors("Es ist ein Fehler aufgetreten");
        }

        foreach($articleNumbers as $key => $articleNr)
        {
            $article = new Article();
            $article->title = $articleNr;
            $article->project_id = $project->id;
            $article->save();
            
            foreach($articleVariables as $varKey => $variable)
            {
                if($varKey == 0 || !isset($variable[$key]) || !isset($tags[$varKey]) || $tags[$varKey] == -1) continue;
                $articleVariable = new ArticleVariable();
                $articleVariable->article_id = $article->id;
                $articleVariable->variable_id = $varKey;
                $articleVariable->value = $variable[$key];
                $articleVariable->save();
            }
        }

        ProjectInitializationJob::dispatch($project)->delay(now()->addSeconds(20));

        return redirect()->route('index');
    }



    /**
     * Show the form for editing the specified resource.
     *
     */
    public function edit($projectId) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("editProjects", $loggedUser))
            return redirect()->route('index');


        # Projektdaten lesen
        $project = Project::find($projectId);
        if ($project == null)
            return redirect()->route("projects.list")->withErrors("Projekt nicht gefunden.");

        # Darf nur Aufträge sehen, bei denen verantwortlich -> Prüfen, ob verantwortlich
//        if (UserController::checkRight("seeOnlyOwnOrders", $loggedUser) && !(UserController::isResponsibleOperatorForOrder($loggedUser->operator, $project)))
//            return redirect()->route('index');

        $client = $loggedUser->client;

        # Für eventuelle Änderungen weitere Auftragskomponenten laden
        # Auftraggeberliste laden
        $customerList = Customer::all();

        # Projekttypen laden
        $projectTypeList = ProjectType::all();

        $articleList = $project->articles;


        # Auftragshistorie lesen
//        $orderLogs = array();
//        if (count($order->order_logs) > 0) {
//
//            # 14.01.2019 / DS / Wenn sensible Daten ausgeblendet werden sollen, nur Freitext-Lognachrichten anzeigen und
//            # die Systemlogs, die durch den Nutzer selbst getriggert wurden
//            if (UserController::checkRight("hideSensibleOrderData", $loggedUser)) {
//                $customLogs = $order->order_custom_logs()->get();
//                $ownLogs = $order->order_logs()->where('operator_id', '=', $loggedUser->operator->id)->get();
//
//                $logs = $customLogs->merge($ownLogs)->sortByDesc("created_at");
//            }
//            # Ansonsten alle Lognachrichten anzeigen
//            else $logs = $order->order_logs()->orderBy("created_at", "desc")->get();
//
//            foreach ($logs as $order_log_id => $order_log) {
//
//                # Zeitstempel in lesbare Form bringen
//                $timeObject = \DateTime::createFromFormat("Y-m-d H:i:s", $order_log->created);
//                if (!$timeObject) continue;
//
//                $orderLogs[$order_log_id]["created"] = $timeObject->format("d.m.Y");
//                $orderLogs[$order_log_id]["type"] = $order_log->type;
//                $orderLogs[$order_log_id]["text"] = $order_log->text;
//
//                # Systemlogs werden extra gekennzeichnet, Standardeinstellung
//                $orderLogs[$order_log_id]["creator"] = "System";
//                $orderLogs[$order_log_id]["class"] = "systemlog";
//
//                # Wenn Logeintrag kein Systemlog, anderweitig kennzeichnen und Nutzernamen auslesen
//                if ($order_log->type != OrderLogs::LOGTYPE_SYSLOG) {
//                    # Nutzernamen holen
//                    $operator = Operator::find($order_log->operator_id);
//                    if ($operator == null) continue;
//                    $orderLogs[$order_log_id]["creator"] = $operator->name . " " . $operator->firstname;
//                    $orderLogs[$order_log_id]["class"] = "userlog";
//                }
//            }
//        }



        # Verantwortliche Bearbeiter des Auftrags laden
        $responsibleOperatorsIds = $project->project_operator_responsible->pluck('id')->toArray();
        $responsibleOperators = $project->project_operator_responsible;

        # Alle weiteren Bearbeiter laden
//        $operators = $client->operatorsWithUser->whereNotIn("id", $responsibleOperatorsIds);
        $operators = Operator::all()->whereNotIn("id", $responsibleOperatorsIds);

        # Validierung
        # 14.01.2019 / DS / Wenn der Nutzer die sensiblen Daten nicht sehen darf, sind andere Validierungen nötig,
        # da manche Felder im Frontend dann nicht vorhanden sind.
//        if (UserController::checkRight("hideSensibleOrderData", $loggedUser))
//            $validationRules = $this->orderCreateValidationRules;
//        else $validationRules = $this->orderCreateSensibleValidationRules;
//        $frontValidator = JsValidator::make($validationRules, [], [], "#editOrder");

        $parameters = ["orderToEdit" => $project,
            "customerList" => $customerList,
            "orderTypesList" => $projectTypeList,
            "articleList" => $articleList,
//            "orderTagsList" => $orderTagsList,
//            "orderStatusList" => $orderStatusList,
//            "orderPaymentStatusList" => $orderPaymentStatusList,
//            "orderAddresses" => $orderAddresses,
//            "orderDocuments" => $orderDocuments,
//            "orderLogs" => $orderLogs,
//            "orderAppointments" => $orderAppointments,
//            "orderMailLogs" => $orderMailLogs,
//            "orderFollowUps" => $orderFollowUps,
            "responsibleOperatorsList" => $responsibleOperators,
            "operatorsList" => $operators,
//            "orderCreditNotes" => $orderCreditNotes,
//            "validator" => $frontValidator,
//            "orderInvoices" => $orderInvoices
        ];

        return View::make('project.edit')->with($parameters);
    }





    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     */
    public function update(Request $request) {

        $loggedUser = User::find(Auth::user()->id);

        $projectToEdit = Project::find($request->get("orderToEdit"));

        # Berechtigung prüfen
        if (!UserController::checkRight("editProjects", $loggedUser))
            return redirect()->route('index');

        # Darf nur Projekte bearbeiten, bei denen verantwortlich -> Prüfen, ob verantwortlich
//        if (UserController::checkRight("seeOnlyOwnOrders", $loggedUser) && !(UserController::isResponsibleOperatorForOrder($loggedUser->operator, $orderToEdit)))
//            return redirect()->route('index');


        $client = $loggedUser->client;

        # 14.01.2019 / DS / Wenn der Nutzer die sensiblen Daten nicht sehen darf, sind andere Validierungen nötig,
        # da manche Felder im Frontend dann nicht vorhanden sind.
//        if (UserController::checkRight("hideSensibleProjectData", $loggedUser))
//            $validationRules = $this->orderCreateValidationRules;
//        else $validationRules = $this->orderCreateSensibleValidationRules;
//
//        # Input validieren, soweit möglich
//        $backValidator = Validator::make($request->all(), $validationRules);
//
//        if ($backValidator->fails()) {
//            return redirect()->back()->withErrors($backValidator->errors());
//        }


        # Errorflags
        $responsibleOperatorsError = false;
        $orderTagsError = false;
        $orderAddressError = false;
        $orderCreditNoteError = false;
        $orderFilesError = false;
        $orderFilesDeleteError = false;
        $orderAppointmentError = false;
        $orderLogError = false;
        $orderValueError = false;
        $orderEndDateError = false;
        $orderExportDateError = false;


        # Bei grundlegenden Projektdaten überprüfen, ob etwas geändert wurde,
        # wenn ja, Auftragshistorie-Eintrag schreiben.


        # 14.01.2019 / DS / Sensible Daten nur ändern / auf Änderung überprüfen, wenn Nutzer das entsprechende Recht
        # hat (hideSensibleOrderData == false)
        if (!UserController::checkRight("hideSensibleProjectData", $loggedUser)) {

            # Auftraggeber
            $newCustomerID = $request->get("customer_id");

            if ($projectToEdit->customer_id != $newCustomerID) {
                # Namen des alten Auftraggebers holen
                $oldCustomer = $projectToEdit->customer->company;

                # Namen des neuen Auftraggebers holen
                $newCustomer = Customer::find($newCustomerID);

                # Wenn neuer Auftraggeber nicht gefunden wird, Fehler zurückgeben: Kein Auftrag ohne gültigen Auftraggeber!
                if ($newCustomer == null) return redirect()->back()->withErrors("Neuen Auftraggeber nicht gefunden!");

                $newCustomer = $newCustomer->company;

                # Log schreiben
                OrderLogsController::writeLog(2, $projectToEdit->id, $loggedUser->operator->id,
                    ["{OLDCUSTOMER}" => $oldCustomer, "{NEWCUSTOMER}" => $newCustomer]);

                $projectToEdit->customer_id = $newCustomerID;
            }

            # Projekttyp
            $newOrderTypeID = $request->get("project_type_id");

            if ($projectToEdit->order_type_id != $newOrderTypeID) {
                # Alten Projekttyp lesen
                $oldProjectType = $projectToEdit->project_type->title;

                # Neuen Projekttyp lesen
                $newProjectType = ProjectType::find($newOrderTypeID);

                # Wenn neuer Projekttyp nicht gefunden wird, Fehler zurückgeben: Kein Projekt ohne gültigen Projekttyp!
                if ($newProjectType == null) return redirect()->back()->withErrors("Neuen Projekttyp nicht gefunden!");

                $newProjectType = $newProjectType->title;

                # Log schreiben
                OrderLogsController::writeLog(3, $projectToEdit->id, $loggedUser->operator->id,
                    ["{OLDPROJECTTYPE}" => $oldProjectType, "{NEWPROJECTTYPE}" => $newProjectType]);

                $projectToEdit->project_type_id = $newOrderTypeID;
            }
        }


        # Zwischenspeichern
        try {
            $projectToEdit->save();
        } catch (\Exception $e) {
            return redirect()->back()->withErrors("Speichern fehlgeschlagen.");
        }


        # 14.01.2019 / DS
        if (!UserController::checkRight("hideSensibleProjectData", $loggedUser)) {

            # Verantwortlicher Bearbeiter
            # Bisherige verantwortliche Bearbeiter holen
            $responsibleOperatorsIds = $projectToEdit->project_operator_responsible->pluck('id')->toArray();

            $newResponsibleOperators = $request->get("responsibleOperators");

            if ($newResponsibleOperators <> null)
                $newResponsibleOperators = array_keys($newResponsibleOperators);
            else $newResponsibleOperators = array();

            $operatorsToAdd = array();

            foreach ($newResponsibleOperators as $operatorId) {

                $addOperator = Operator::find($operatorId);
                if ($addOperator == null) continue;

                $operatorsToAdd[] = $addOperator->id;
            }

            try {
                $projectToEdit->project_operator_responsible()->sync($operatorsToAdd);
            } catch (\Exception $e) {
                Log::error("Fehler beim Hinzufügen von verantwortlichen Bearbeitern: " . $e->getMessage());
                $responsibleOperatorsError = true;
            }

            # Für Log prüfen, ob etwas geändert wurde
            #
            $newOperatorCount = count($newResponsibleOperators);
            $oldOperatorCount = count($responsibleOperatorsIds);

            $logOperatorChange = false;
            if ($oldOperatorCount == $newOperatorCount) {
                # Differenz der Arrays
                if (count(array_diff($responsibleOperatorsIds, $newResponsibleOperators)) > 0)
                    $logOperatorChange = true;
            }
            else $logOperatorChange = true;

            if ($logOperatorChange)
                OrderLogsController::writeLog(26, $projectToEdit->id, $loggedUser->operator->id);
//
//            # Log schreiben, wenn etwas geändert wurde
//            if ($request->get("orderTagsChanged") == true) {
//                OrderLogsController::writeLog(6, $orderToEdit->id, $loggedUser->operator->id);
//            }
        }








        ### Auftragsdokumente zuordnen ###
        $projectDocumentRepository = new ProjectDocumentRepository();
        $projectDocumentRepository->dropzoneName = "newProjectFiles";
        $projectDocumentRepository->resolveDocuments($loggedUser, $request);
//        $projectType = $projectDocumentRepository->projectType;

        //Create a new Tag instance (fill the array with your own database fields)

//In the tag relationship, save a new video
//        $projectToEdit->documents()->save($newDocument);


//        $projectTypeDocumentRepository = new ProjectTypeDocumentRepository();
//        $projectTypeDocumentRepository->dropzoneName = "newArticleFiles";
//        $projectTypeDocumentRepository->setTemplateType("article");
//        $projectTypeDocumentRepository->initializeDocuments($loggedUser, $request);
//        $projectType = $projectTypeDocumentRepository->projectType;









//        # Vom Bearbeiter verfasste Historieneinträge schreiben
//        $customOrderLogs = json_decode($request->get("newLogEntries"));
//
//        if ($customOrderLogs == NULL) {
//            Log::error("Fehler beim Hinzufügen von Auftragshistorie-Einträgen; JSON-Fehler: " . json_last_error_msg());
//            $orderLogError = true;
//        }
//        else {
//            if (count($customOrderLogs->logs) > 0) {
//                foreach ($customOrderLogs->logs as $customOrderLog) {
//                    if (!OrderLogsController::writeCustomLog($customOrderLog->text, $customOrderLog->order_id, $customOrderLog->operator_id)) {
//                        $orderLogError = true;
//                    }
//                }
//            }
//        }

        # Erfolgsmeldung zurückgeben, ggf. mit Hinweisen
        $message = "Das Projekt " . $projectToEdit->project_nr . " wurde erfolgreich gespeichert. ";
        if ($orderTagsError) $message .= "Es gab einen Fehler beim Aktualisieren der Auftragskennzeichen.";
        if ($orderAddressError) $message .= "Es gab einen Fehler beim Aktualisieren der Auftragsadressen.";
        if ($orderFilesError) $message .= "Es gab einen Fehler beim Hinzufügen der Auftragsdokumente.";
        if ($orderFilesDeleteError) $message .= "Es gab einen Fehler beim Löschen von Auftragsdokumente.";
        if ($orderLogError) $message .= "Es gab einen Fehler beim Hinzufügen von Einträgen in der Auftragshistorie.";
        if ($orderAppointmentError) $message .= "Es gab einen Fehler beim Aktualisieren eines Auftragstermins.";
        if ($orderValueError) $message .= "Es gab einen Fehler beim Aktualisieren des Auftragswerts.";
        if ($orderEndDateError) $message .= "Es gab einen Fehler beim Setzen des Endtermins.";
        if ($orderExportDateError) $message .= "Es gab einen Fehler beim Setzen des Exporttermins.";
        if ($responsibleOperatorsError) $message .= "Es gab einen Fehler beim Aktualisieren der verantwortlichen Bearbeiter.";
        if ($orderCreditNoteError) $message .= "Es gab einen Fehler beim Aktualisieren der Gutschriften.";

        if ($orderTagsError || $orderAddressError || $orderFilesError || $orderFilesDeleteError || $orderLogError ||
            $orderAppointmentError || $orderEndDateError || $responsibleOperatorsError || $orderCreditNoteError || $orderExportDateError)
            return redirect()->route('projects.list')->with('okayWithErrors', $message);
        else
            return redirect()->back()->with('message', $message);

    }



    /**
     * Show the modal to confirm the destroy of a resource.
     *
     */
    public function showDeleteConfirm() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("deleteProjects", $loggedUser))
            return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

        $projectId = request()->input('projectId');

        # Projektdaten lesen
        $project = Project::find($projectId);
        if ($project == null)
            return response()->json(['response' => 'error', 'message' => 'Projekt nicht gefunden.']);

        # Darf nur auf Aufträge zugreifen, bei denen verantwortlich -> Prüfen, ob verantwortlich
//        if (UserController::checkRight("seeOnlyOwnProjects", $loggedUser) && !(UserController::isResponsibleOperatorForOrder($loggedUser->operator, $order)))
//            return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);


        # Kunde
        if ($project->customer <> null)
            $project->customerName = $project->customer->company . " " . $project->customer->place;
        else
            $project->customerName = "Unbekannt";

        # Projekttyp
        if ($project->project_type <> null)
            $project->project_type_title = $project->project_type->title;
        else
            $project->project_type_title = "Unbekannt";

        # View rendern
        $arguments = ["project" => $project];

        $editorHTML = View::make('project.deleteProject')->with($arguments)->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }



    /**
     * Remove the specified resource from storage.
     *
     */
    public function destroy() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("deleteProjects", $loggedUser))
            return redirect()->route('index');

        $projectId = request()->input('projectToDelete');
        $confirmDelete = request()->input('confirmDelete');

        # Projektdaten lesen
        $project = Project::find($projectId);
        if ($project == null)
            return redirect()->back()->withErrors("Projekt nicht gefunden.");

        # Darf nur auf Aufträge zugreifen, bei denen verantwortlich -> Prüfen, ob verantwortlich
//        if (UserController::checkRight("seeOnlyOwnProjects", $loggedUser) && !(UserController::isResponsibleOperatorForOrder($loggedUser->operator, $order)))
//            return redirect()->back()->withErrors("Keine Berechtigung.");

        # Prüfen, ob Checkbox bestätigt ist
        if (!is_null($confirmDelete)) {
            # Projekt löschen - wenn möglich!
            try {
                # Versuchen, zu löschen
                $project->delete();
            } catch (\Exception $e) {
                # Es kann sein, dass noch Relations zu Projekten (projects) bestehen
                Log::error("Fehler beim Löschen des Projekts " . $project->id . ", Mandant " .
                    $loggedUser->client->id . ": " . $e->getMessage() . $e->getFile() . $e->getLine());

                return redirect()->back()->withErrors("Projekt konnte nicht gelöscht werden.");
            }
        }
        else {
            return redirect()->back()->with("message", "Der Projekt wurde nicht gelöscht, da das Löschen bestätigt werden muss!");
        }

        Log::info("Projekt " . $project->id . ", Mandant " . $loggedUser->client->id . " wurde gelöscht.
                            (User " . $loggedUser->id . ")");
        return redirect()->route('projects.list')->with("message", "Projekt wurde erfolgreich gelöscht.");
    }









}