<?php

namespace App\Http\Controllers;

use App\OrderStatus;
use App\OrderTypes;
use App\Variable;
use Illuminate\Http\Request;

use App\User;
use App\Client;
use App\ProjectType;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use JsValidator;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

class VariablesController extends Controller {

    private $variableValidationRules = [
        'title' => 'required|string|max:255',
    ];

    /**
     * Display a listing of the resource.
     */
    public function indexArticleVariables() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("seeArticleVariables", $loggedUser))
            return redirect()->route('index');


        $variableList = Variable::all()->where("ref_type", "=", "article");
        return View::make('variables.articleVariables.list')->with(["variableList" => $variableList]);
    }

    public function indexProjectVariables() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("seeArticleVariables", $loggedUser))
            return redirect()->route('index');


        $variableList = Variable::all()->where("ref_type", "=", "project");
        return View::make('variables.projectVariables.list')->with(["variableList" => $variableList]);
    }

    /**
     * Show the form for creating a new resource.
     * Show the form for editing the specified resource.
     *
     */
    public function edit() {
        $loggedUser = User::find(Auth::user()->id);

        $variableId = request()->input('variableId');
        $ref_type = request()->input('ref_type');

        # Neuen Auftragstyp erstellen
        if ($variableId == 0) {
            # Berechtigung prüfen
            if (!UserController::checkRight("createVariables", $loggedUser))
                return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

            $variable = new Variable();
            $variable->ref_type = $ref_type;
        } # Auftragstyp bearbeiten
        else {
            # Berechtigung prüfen
            if (!UserController::checkRight("editVariables", $loggedUser))
                return response()->json(['response' => 'error', 'message' => ' Keine Berechtigung.']);

            # Auftragstyp finden
            $variable = Variable::find($variableId);
            $variable->ref_type = $ref_type;
            if ($variable == null)
                return response()->json(['response' => 'error', 'message' => 'Variable nicht gefunden.']);

        }

        # Auftragsstatus auslesen (für Belegung Startstatus)
//        $orderStatusList = $loggedUser->client->order_status;

        # Validierung
        $frontValidator = JsValidator::make($this->variableValidationRules, [], [], "#editVariable");

        $arguments = ["variable" => $variable,
            "validator" => $frontValidator,
//            "orderStatusList" => $orderStatusList
        ];

        # View rendern
        $editorHTML = View::make('variables.articleVariables.edit')->with($arguments)->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }


    /**
     * Store a newly created resource
     *
     */
    public function store(Request $request) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("createProjectTypes", $loggedUser))
            return redirect()->route('index');

        # Speichern
        $trySave = $this->save($request, $loggedUser->client);
        if ($trySave instanceof Variable) {

            if($trySave->ref_type == "project") {
                return redirect()->route('projectVariables.list')->with("message", "Variable " . $trySave->title . " wurde erfolgreich angelegt.");
            }
            else {
                return redirect()->route('articleVariables.list')->with("message", "Variable " . $trySave->title . " wurde erfolgreich angelegt.");
            }
        }
        # Wenn Speichern nicht erfolgreich, enthält $trySave eventuelle Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            return redirect()->back()->withErrors($trySave);
        }
    }


    /**
     * Update the specified resource
     *
     */
    public function update(Request $request) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("editProjectTypes", $loggedUser))
            return redirect()->route('index');

        # Auftragstyp lesen
        $variable = Variable::find($request->get("variableId"));
        if ($variable == null) {
            return redirect()->back()->withErrors("Projekttyp nicht gefunden.");
        }

        $variable->title = $request->get('title');
        $variable->variable_title = $request->get('variable_title');
        $variable->type = $request->get('variable_type');
        $variable->ref_type = $request->get('ref_type');
        $variable->static = 0;

        # Speichern
        $trySave = $this->save($request, $loggedUser->client, $variable);
        if ($trySave instanceof Variable) {
            if($variable->ref_type == "project") {
                return redirect()->route('projectVariables.list')->with("message", "Variable " . $variable->title . " wurde erfolgreich gespeichert.");
            }
            else{
                return redirect()->route('articleVariables.list')->with("message", "Variable " . $variable->title . " wurde erfolgreich gespeichert.");
            }
        }
        # Wenn Speichern nicht erfolgreich, enthält $trySave eventuelle Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            return redirect()->back()->withErrors($trySave);
        }
    }


    /**
     * Save the specified resource in storage.
     *
     */
    public function save(Request $request, Client $client, Variable $variable = null) {
        if (is_null($variable)) $variable = new Variable();

        # Input validieren
        $backValidator = Validator::make($request->all(), $this->variableValidationRules);

        if ($backValidator->fails()) {
            return $backValidator->errors();
        }

        # Neue Daten setzen
//        $orderType->client_id = $client->id;
        $variable->title = $request->get('title');
        $variable->variable_title = $request->get('variable_title');
        $variable->type = $request->get('variable_type');
        $variable->ref_type = $request->get('ref_type');
        $variable->static = 0;
//        $orderType->order_statuses_id = $request->get('order_statuses_id');

        # Speichern
        try {
            $variable->save();
        } catch (\Exception $e) {
            Log::error("Fehler beim Speichern der Daten für Variable " . $variable->title . ", (ID: " . $variable->id . " ), Mandant " .
                $client->id . ": " . $e->getMessage());
            return "Es ist ein Fehler aufgetreten.";
        }

        return $variable;
    }


    public function showDeleteConfirm() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("deleteVariables", $loggedUser))
            return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

        $variableId = request()->input('variableId');

        # Auftragstyp lesen
        $variable = Variable::find($variableId);
        if ($variable == null)
            return response()->json(['response' => 'error', 'message' => 'Variable nicht gefunden.']);

        $arguments = ["confItem" => "variables",
            "confItemLabel" => "Variable",
            "confItemLabelArticle" => "die",
            "additional_info" => "",
            "item" => $variable];

        # View rendern
        $editorHTML = View::make('project_conf.delete')->with($arguments)->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }

    /**
     * Remove the specified resource from storage.
     *
     */
    public function destroy() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::isClientAdmin($loggedUser) && !UserController::checkRight("deleteVariables", $loggedUser))
            return redirect()->route('index');

        $variableId = request()->input('variablesId');
        $confirmDelete = request()->input('confirmDelete');

        # Auftragstyp lesen
        $variable = Variable::find($variableId);
        if ($variable == null)
            return redirect()->back()->withErrors("Variable nicht gefunden.");

        # Prüfen, ob Checkbox bestätigt ist
        if (!is_null($confirmDelete)) {
            # Auftragstyp löschen - wenn möglich!
            try {
                # Versuchen, zu löschen
                $variable->delete();
            } catch (\Exception $e) {
                # Es kann sein, dass noch Relations zu Auftragskennzeichen (order_tags) oder Aufträgen (orders) bestehen
                Log::error("Fehler beim Löschen der Variable " . $variable->id . ", Mandant " .
                    $loggedUser->client->id . ": " . $e->getMessage());

                return redirect()->back()->withErrors("Variable konnte nicht gelöscht werden. Es bestehen noch 
            Verbindungen zu Dokumenten.");
            }
        }
        else {
            return redirect()->back()->with("message", "Die Variable wurde nicht gelöscht, da das Löschen bestätigt werden muss!");
        }

        Log::info("Projekttyp " . $variable->id . ", Mandant " . $loggedUser->client->id . " wurde gelöscht. 
                            (User " . $loggedUser->id . ")");

        if($variable->ref_type == "project") {
            return redirect()->route('projectVariables.list')->with("message", "Variable wurde erfolgreich gelöscht.");
        }
        else {
            return redirect()->route('articleVariables.list')->with("message", "Variable wurde erfolgreich gelöscht.");
        }
    }
}
