<?php

namespace App\Http\Controllers;

use App\User;
use App\Client;
use App\Customer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use JsValidator;

class CustomerController extends Controller {

    private $customerValidationRules = [
        'company' => 'required|string|max:255',
        # toDo: Validierungen vorübergehend auskommentiert, da Daten von PK nicht vollständig
       # 'street' => 'required|regex:/\b[A-Za-z\.\-\säÄöÖüÜß]+/|max:255',
       # 'house_number' => 'required|regex:/\b[0-9]{1,4}[a-zA-Z]{0,1}[-]?[0-9]{0,4}[a-zA-Z]{0,1}?\b/', # lässt auch Hausnummernbereiche, z.B. 8-12 zu
       # 'zipcode' => 'required|regex:/\b[0-9]{4,6}\b/',
       # 'place' => 'required|regex:/\b[A-Za-z\-\säÄöÖüÜß]+\b/|max:255',
       # 'country' => 'required|regex:/\b[A-Za-zß\-\s]+\b/|max:255',

       # 'name' => 'required|string|max:255',
       # 'firstname' => 'required|string|max:255',
       # 'telephone_number' => 'required|string|max:30',
       # 'fax_number' => 'string|nullable|max:30',
       # 'email' => 'required|email|max:255',
        'pricegroup' => 'required|integer',
    ];

    private $formTitleList = [
        "Keiner" => NULL,
        "Frau" => "Frau",
        "Herr" => "Herr"
    ];

    /**
     * Display a listing of the resource.
     */
    public function index() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("seeCustomersList", $loggedUser))
            return redirect()->route('index');

        # Auftraggeber des Mandanten holen
        $customerList = $loggedUser->client->customer;

        return View::make('customer.list')->with(["customerList" => $customerList]);
    }

    /**
     * Show the form for creating a new resource.
     * Show the form for editing the specified resource.
     *
     */
    public function edit($customerId) {
        $loggedUser = User::find(Auth::user()->id);

        # Neuen Auftraggeber erstellen
        if ($customerId == 0) {
            # Berechtigung prüfen
            if (!UserController::checkRight("createCustomers", $loggedUser))
                return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

            $customer = new Customer();
        }
        # Auftraggeber bearbeiten
        else {
            # Berechtigung prüfen
            if (!UserController::checkRight("editCustomers", $loggedUser))
                return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

            # Auftraggeber finden
            $customer = Customer::find($customerId);
            if ($customer == null) {
                return response()->json(['response' => 'error', 'message' => 'Auftraggeber nicht gefunden.']);
            }
        }

        # Validierung
        $frontValidator = JsValidator::make($this->customerValidationRules, [], [], "#editCustomer");

        $arguments = ["customer" => $customer,
            "titleList" => $this->formTitleList,
            "validator" => $frontValidator
        ];

        # View rendern
        $editorHTML = View::make('customer.edit')->with($arguments)->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }

    /**
     * Store a newly created resource
     *
     */
    public function store(Request $request) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("createCustomers", $loggedUser))
            return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

        # Auftraggebernummer aus Nummernkreis ziehen
        $newCustomer = new Customer();
        $newCustomer->customer_number = $loggedUser->client->numberRangeCustomers->getNextNumber();

        # Speichern
        $saveCustomer = $this->save($request, $loggedUser->client, $newCustomer);
        if ($saveCustomer instanceof Customer)
            return response()->json(['response' => 'success',
                'message' => 'Auftraggeber ' . $saveCustomer->company . ' wurde erfolgreich angelegt.',
                'newCustomer' => $saveCustomer]);
        # Wenn Speichern nicht erfolgreich, enthält $saveCustomer eventuell Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            return response()->json(['response' => 'error', 'message' => $saveCustomer]);
        }
    }

    /**
     * Update the specified resource
     *
     */
    public function update(Request $request) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("editCustomers", $loggedUser))
            return redirect()->route('index');

        $customerId = $request->get('customer_id');

        # Bestehenden Auftraggeber lesen
        $customer = Customer::find($customerId);

        if ($customer == null)
            return response()->json(['response' => 'error', 'message' => "Auftraggeber nicht gefunden."]);

        # Speichern
        $trySave = $this->save($request, $loggedUser->client, $customer);
        if ($trySave instanceof Customer) {
            return response()->json(['response' => 'success', 'message' => "Auftraggeber " . $customer->company .
                " wurde erfolgreich gespeichert."]);
        }
        # Wenn Speichern nicht erfolgreich, enthält $trySave eventuell Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            return response()->json(['response' => 'error', 'message' => $trySave]);
        }
    }

    /**
     * Save the specified resource in storage.
     *
     */
    public function save(Request $request, Client $client, Customer $customer = null) {
        if (is_null($customer)) $customer = new Customer();

        # Input validieren
        $backValidator = Validator::make($request->all(), $this->customerValidationRules);

        if ($backValidator->fails()) {
            return $backValidator->errors();
        }

        # Neue Daten setzen
        $customer->client_id = $client->id;
        $customer->company = $request->get('company');
        $customer->title = $request->get('title');
        $customer->name = $request->get('name');
        $customer->firstname = $request->get('firstname');
        $customer->street = $request->get('street');
        $customer->house_number = $request->get('house_number');
        $customer->zipcode = $request->get('zipcode');
        $customer->place = $request->get('place');
        $customer->country = $request->get('country');
        $customer->telephone_number = $request->get('telephone_number');
        $customer->fax_number = $request->get('fax_number');
        $customer->email = $request->get('email');
        $customer->pricegroup = $request->get('pricegroup');

        # Speichern
        try {
            $customer->save();
        } catch (\Exception $e) {
            Log::error("Fehler beim Speichern der Daten für Auftraggeber " . $customer->label . ", (ID: " . $customer->id . " ), Mandant " .
                $client->id . ": " . $e->getMessage());
            return ["Es ist ein Fehler aufgetreten."];
        }

        return $customer;
    }

    /**
     * Remove the specified resource from storage.
     *
     */
    # Derzeit nicht erreichbar
    public function destroy($customerId) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("deleteCustomers", $loggedUser))
            return redirect()->route('index');

        # Auftraggeber lesen
        $customer = Customer::find($customerId);
        if ($customer == null) {
            return redirect()->back()->withErrors("Auftraggeber nicht gefunden.");
        }

        # Auftraggeber löschen - wenn möglich!
        try {
            # Versuchen, zu löschen
            $customer->delete();
        } catch (\Exception $e) {
            # Es kann sein, dass noch Relations zu Aufträgen (orders) bestehen
            Log::error("Fehler beim Löschen des Auftraggebers " . $customer->id . ", Mandant " .
                $loggedUser->client->id . ": " . $e->getMessage());

            return redirect()->back()->withErrors("Auftraggeber konnte nicht gelöscht werden. Es bestehen noch
            Verbindungen zu Aufträgen.");
        }

        Log::info("Auftraggeber " . $customer->id . ", Mandant " . $loggedUser->client->id . " wurde gelöscht.
                            (User " . $loggedUser->id . ")");

        return redirect()->route('customer.list')->with("message", "Auftraggeber wurde erfolgreich gelöscht.");
    }


    public function asyncSearch($search = "") {
        $loggedUser = User::find(Auth::user()->id);

        # Nach Auftraggebern suchen, die in etwa dem Namen entsprechen
        $possibleCustomers = Customer::where('company', 'like', '%' . $search . '%')
            ->where('client_id', '=', $loggedUser->client->id)
            ->pluck('id', 'company')->toArray();

        if (count($possibleCustomers) < 0) {
            $possibleCustomers = false;
        }

        return json_encode($possibleCustomers);
    }
}
