<?php

namespace App\Http\Controllers;

use App\OrderAppointmentOperator;
use App\Textblock;
use App\User;
use App\Client;
use App\Orders;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Log;
use JsValidator;
use Illuminate\Support\Facades\Validator;

use App\Mail\OrderMail;
use App\OrderMailLog;
use Illuminate\Http\Request;

use App\Jobs\SendOrderMailJob;
use Illuminate\Support\Facades\Storage;

class OrderMailController extends Controller {
    private $orderMailValidationRules = [
        "to" => "required|email|max:255",
        "subject" => "required|string|max:255",
        "text" => "required|string"];

    public function create($orderId, $appointmentId = 0) {
        $loggedUser = User::find(Auth::user()->id);

        $arguments = array();

        # Berechtigung prüfen
        if (!UserController::checkRight("editOrders", $loggedUser))
            return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

        # Bearbeiter-ID des Nutzers holen
        $operatorId = $loggedUser->operator->id;

        # Auftragsdaten holen
        $order = Orders::find($orderId);
        if ($order == null)
            return response()->json(['response' => 'error', 'message' => 'Auftrag nicht gefunden.']);

        # Auftragsdokumente holen
        $orderDocuments = OrdersController::getOrderDocumentsList($order->id);

        # Für die Vorblendung im View Namen des Bearbeiters holen
        $operatorLabel = $loggedUser->operator->getName();

        # Normale Textbausteine für E-Mails holen
        $mailTemplates = $loggedUser->client->textblocksOrderMail;

        # Empfänger
        $mailTo = $order->customer->email;

        # Wenn ein Termin versendet werden soll, Textbaustein holen und Platzhalter ersetzen
        if ($appointmentId <> 0) {
            $appointmentTextBlock = $loggedUser->client->textblockAppointmentMail;

            # Termindaten holen
            $appointment = OrderAppointmentOperator::find($appointmentId);

            $tmpTextblock = $this->makeAppointmentMailTextblock($appointmentTextBlock, $appointment, $loggedUser);

            $mailTemplates->add($tmpTextblock);
            $arguments["selectBlock"] = $tmpTextblock->id;

            # Wenn Terminversand, dann wird als Empfänger die E-Mail-Adresse der ersten Auftragsadresse vorgeblendet
            $firstAddress = $order->addresses->first();
            if ($firstAddress <> NULL && $firstAddress->email_final_customer <> "")
                $mailTo = $firstAddress->email_final_customer;
        }

        # Validierung
        $frontValidator = JsValidator::make($this->orderMailValidationRules, [], [], "#addOrderMail");

        $arguments = array_merge($arguments, ["order" => $order,
            "operatorID" => $operatorId,
            "operatorLabel" => $operatorLabel,
            "mailTemplates" => $mailTemplates,
            "orderDocuments" => $orderDocuments,
            "mailTo" => $mailTo,
            "validator" => $frontValidator]);

        # View rendern
        $editorHTML = View::make('order_mails.create')->with($arguments)->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }


    private function makeAppointmentMailTextblock($appointmentTextBlock, $appointment, $loggedUser) {

        # Wenn Auftragsadresse vorhanden, aus dieser Namen für E-Mail ziehen
        # Erste Adresse des Auftrags lesen
        $firstAddress = $appointment->order->addresses->first();
        if ($firstAddress <> NULL && $firstAddress->email_final_customer <> "") {
            $title = "Sehr geehrte/r Herr/Frau " . $firstAddress->name;
        }

        # Ansonsten Kundenkontakt nutzen
        else {

            # Anrede bauen
            $title = "Sehr ";
            if ($appointment->order->customer->title == "Frau")
                $title .= "geehrte ";
            else
                $title .= "geehrter ";

            $title .= $appointment->order->customer->title . " " . $appointment->order->customer->name;
        }

        # Platzhalter
        $placeholders["##ANREDE##"] = $title;

        $prettyDate = Carbon::createFromTimeString($appointment->appointment_from);
        $placeholders["##TERMINDATUM##"] = $prettyDate->format("d.m.Y");
        $placeholders["##BEARBEITER##"] = $appointment->operator->getName();
        $placeholders["##VON_VORNAME##"] = $loggedUser->firstname;
        $placeholders["##VON_NACHNAME##"] = $loggedUser->name;

        $tmpTextblock = new Textblock();

        # Temporäre ID für Vorselektion des Textbausteins
        $tmpTextblock->id = $tmpTextblock->getLastId() + 1;
        # Betreff übernehmen
        $tmpTextblock->subject = $appointmentTextBlock->subject;
        # Text erstellen
        $tmpTextblock->text = TextblockController::makeText($appointmentTextBlock->text, $placeholders);

        return $tmpTextblock;
    }

    # Hinzufügen einer Auftragsmail über AJAX-Request
    public function asyncAddOrderMail() {
        // Inhalt der Mail holen und dekodieren
        $newMailJson = request()->get("newMail");

        $newMail = json_decode($newMailJson);

        # Input validieren
        $backValidator = Validator::make(json_decode(json_encode($newMail), true), $this->orderMailValidationRules);

        if ($backValidator->fails()) {
            return response()->json(['response' => 'error', 'message' => $backValidator->errors()]);
        }

        if ($newMail == null) {
            return response()->json(['response' => 'error', 'message' => 'Die E-Mail konnte nicht hinterlegt werden.']);
        }

        // Mail zu Warteschlange hinzufügen
        $queueMail = $this->sendOrderMail($newMail->to, $newMail->subject, $newMail->text, $newMail->operator_id, $newMail->order_id, $newMail->attachments);

        if ($queueMail) {
            // Auftragslog schreiben
            OrderLogsController::writeLog(14, $newMail->order_id, $newMail->operator_id,
                ["{MAILTO}" => $newMail->to, "{SUBJECT}" => $newMail->subject]);

            return response()->json(['response' => 'success']);
        }
        else {
            return response()->json(['response' => 'error', 'message' => 'Die E-Mail konnte nicht hinterlegt werden.']);
        }
    }

    # Auftragsmail zu Warteschlange hinzufügen
    public function sendOrderMail($to, $subject, $text, $operatorId, $orderId, $attachments = []) {

        # Eintrag für Auftragsmaillog schreiben
        $logEntry = new OrderMailLog();

        $logEntry->order_id = $orderId;
        $logEntry->operator_id = $operatorId;
        $logEntry->to = $to;
        $logEntry->subject = $subject;
        $logEntry->text = $text;
        $logEntry->sent = false;

        try {
            $logEntry->save();
        } catch (\Exception $e) {
            Log::error("Fehler beim Schreiben von Auftragsmaillog: " . $e->getMessage());
            return false;
        }

        $mail = new OrderMail($subject, $text, $attachments);

        $job = new SendOrderMailJob($to, $mail, $operatorId, $orderId, $logEntry->id);

        try {
            dispatch($job)->onQueue('emails');
        } catch (\Exception $e) {
            Log::error("Fehler beim Auftragsmailversand: " . $e->getMessage());
            return false;
        }

        return true;
    }

}
