<?php

namespace App\Http\Controllers;

use App\OrderStatus;
use Illuminate\Http\Request;

use App\User;
use App\Client;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use JsValidator;


class OrderStatusController extends Controller {

    private $orderStatusValidationRules = [
        'label' => 'required|string|max:255',
    ];

    /**
     * Display a listing of the resource.
     */
    public function index() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("seeOrderStatus", $loggedUser))
            return redirect()->route('index');

        # Auftragsstatus des Mandanten auslesen
        $orderStatus = $loggedUser->client->order_status;

        return View::make('order_conf.order_status.list')->with(["orderStatus" => $orderStatus]);
    }

    /**
     * Show the form for creating a new resource.
     * Show the form for editing the specified resource.
     *
     */
    public function edit() {
        $loggedUser = User::find(Auth::user()->id);

        $orderStatusId = request()->input('orderStatusId');

        # Neuen Auftragsstatus erstellen
        if ($orderStatusId == 0) {
            # Berechtigung prüfen
            if (!UserController::checkRight("createOrderStatus", $loggedUser))
                return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

            $orderStatus = new OrderStatus();
        }
        # Auftragsstatus bearbeiten
        else {
            # Berechtigung prüfen
            if (!UserController::checkRight("editOrderStatus", $loggedUser))
                return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

            # Auftragsstatus finden
            $orderStatus = OrderStatus::find($orderStatusId);
            if ($orderStatus == null)
                return response()->json(['response' => 'error', 'message' => 'Auftragsstatus nicht gefunden.']);
        }

        # Validierung
        $frontValidator = JsValidator::make($this->orderStatusValidationRules, [], [], "#editOrderStatus");

        # View rendern
        $editorHTML = View::make('order_conf.order_status.edit')->with(["orderStatus" => $orderStatus, "validator" => $frontValidator])->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }


    /**
     * Store a newly created resource
     *
     */
    public function store(Request $request) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("createOrderStatus", $loggedUser))
            return redirect()->route('index');

        # Speichern
        $trySave = $this->save($request, $loggedUser->client);
        if ($trySave instanceof OrderStatus)
            return redirect()->route('orderStatus.list')->with("message", "Auftragsstatus " . $trySave->label . " wurde erfolgreich angelegt.");
        # Wenn Speichern nicht erfolgreich, enthält $trySave eventuelle Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            return redirect()->back()->withErrors($trySave);
        }
    }

    /**
     * Update the specified resource
     *
     */
    public function update(Request $request, $orderStatusId) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("editOrderStatus", $loggedUser))
            return redirect()->route('index');

        # Auftragsstatus lesen
        $orderStatus = OrderStatus::find($orderStatusId);
        if ($orderStatus == null)
            return redirect()->back()->withErrors("Auftragsstatus nicht gefunden.");


        # Speichern
        $trySave = $this->save($request, $loggedUser->client, $orderStatus);
        if ($trySave instanceof OrderStatus)
            return redirect()->route('orderStatus.list')->with("message", "Auftragsstatus " . $orderStatus->label . " wurde erfolgreich gespeichert.");
        # Wenn Speichern nicht erfolgreich, enthält $trySave eventuelle Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            return redirect()->back()->withErrors($trySave);
        }
    }

    /**
     * Save the specified resource in storage.
     *
     */
    public function save(Request $request, Client $client, OrderStatus $orderStatus = null) {
        if (is_null($orderStatus)) $orderStatus = new OrderStatus();

        # Input validieren
        $backValidator = Validator::make($request->all(), $this->orderStatusValidationRules);

        if ($backValidator->fails()) {
            return $backValidator->errors();
        }

        # Neue Daten setzen
        $orderStatus->client_id = $client->id;
        $orderStatus->label = $request->get('label');

        # Speichern
        try {
            $orderStatus->save();
        } catch (\Exception $e) {
            Log::error("Fehler beim Speichern der Daten für Auftragsstatus " . $orderStatus->label . ", (ID: " . $orderStatus->id . " ), Mandant " .
                $client->id . ": " . $e->getMessage());
            return "Es ist ein Fehler aufgetreten.";
        }

        return $orderStatus;
    }


    public function showDeleteConfirm() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("deleteOrderStatus", $loggedUser))
            return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

        $orderStatusId = request()->input('orderStatusId');

        # Auftragsstatus lesen
        $orderStatus = OrderStatus::find($orderStatusId);
        if ($orderStatus == null)
            return redirect()->back()->withErrors("Auftragsstatus nicht gefunden.");

        $arguments = ["confItem" => "orderStatus",
            "confItemLabel" => "Auftragsstatus",
            "confItemLabelArticle" => "den",
            "item" => $orderStatus];

        # View rendern
        $editorHTML = View::make('order_conf.delete')->with($arguments)->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }

    /**
     * Remove the specified resource from storage.
     *
     */
    public function destroy() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("deleteOrderStatus", $loggedUser))
            return redirect()->route('index');

        $orderStatusId = request()->input('orderStatusId');
        $confirmDelete = request()->input('confirmDelete');

        # Auftragsstatus lesen
        $orderStatus = OrderStatus::find($orderStatusId);
        if ($orderStatus == null)
            return redirect()->back()->withErrors("Auftragsstatus nicht gefunden.");


        # Prüfen, ob Checkbox bestätigt ist
        if (!is_null($confirmDelete)) {
            # Auftragsstatus löschen - wenn möglich!
            try {
                # Versuchen, zu löschen
                $orderStatus->delete();
            } catch (\Exception $e) {
                # Es kann sein, dass noch Relations zu Aufträgen (orders) bestehen
                Log::error("Fehler beim Löschen des Auftragsstatus " . $orderStatus->id . ", Mandant " .
                    $loggedUser->client->id . ": " . $e->getMessage());

                return redirect()->back()->withErrors("Auftragsstatus konnte nicht gelöscht werden. Es bestehen noch 
            Verbindungen zu Aufträgen.");
            }
        }
        else {
            return redirect()->back()->with("message", "Der Auftragsstatus wurde nicht gelöscht, da das Löschen bestätigt werden muss!");
        }

        Log::info("Auftragsstatus " . $orderStatus->id . ", Mandant " . $loggedUser->client->id . " wurde gelöscht. 
                            (User " . $loggedUser->id . ")");
        return redirect()->route('orderStatus.list')->with("message", "Auftragsstatus wurde erfolgreich gelöscht.");
    }
}
