<?php

namespace App\Http\Controllers;

use App\OrderTags;
use App\OrderTypes;
use Illuminate\Http\Request;

use App\User;
use App\Client;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use JsValidator;

class OrderTagsController extends Controller {

    private $orderTagValidationRules = [
        'label' => 'required|string|max:255',
    ];

    /**
     * Display a listing of the resource.
     */
    public function index() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("seeOrderTags", $loggedUser))
            return redirect()->route('index');

        # Auftragskennzeichen des Mandanten auslesen
        $orderTags = $loggedUser->client->order_tags;

        # Für jedes Auftragskennzeichen Label des zugehörigen Auftragstyps auslesen (Für Anzeige)
        # Daten in Array schreiben
        $orderTagsFullData = array();

        foreach ($orderTags as &$orderTag) {
            $oneOrderTag["id"] = $orderTag->id;
            $oneOrderTag["label"] = $orderTag->label;

            $orderType = $orderTag->order_types;

            if ($orderType <> null) {
                $oneOrderTag["order_types_id"] = $orderType->id;
                $oneOrderTag["order_type_label"] = $orderType->label;
            }
            else {
                $oneOrderTag["order_types_id"] = 0;
                $oneOrderTag["order_type_label"] = "nicht zugeordnet";
            }

            $orderTagsFullData[] = $oneOrderTag;
            $oneOrderTag = array();
        }

        return View::make('order_conf.order_tags.list')->with(["orderTags" => $orderTagsFullData]);
    }


    /**
     * Show the form for creating a new resource.
     * Show the form for editing the specified resource.
     *
     */
    public function edit() {
        $loggedUser = User::find(Auth::user()->id);

        $orderTagId = request()->input('orderTagId');

        # Neues Auftragskennzeichen erstellen
        if ($orderTagId == 0) {
            # Berechtigung prüfen
            if (!UserController::checkRight("createOrderTags", $loggedUser))
                return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

            $orderTag = new OrderTags();
            $orderTagOrderType = null;
        }
        # Auftragskennzeichen bearbeiten
        else {
            # Berechtigung prüfen
            if (!UserController::checkRight("editOrderTags", $loggedUser))
                return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

            # Auftragskennzeichen finden
            $orderTag = OrderTags::find($orderTagId);
            if ($orderTag == null)
                return response()->json(['response' => 'error', 'message' => 'Auftragskennzeichen nicht gefunden.']);

            # Zugehörigen Auftragstypen laden
            $orderTagOrderType = $orderTag->order_types;

            # Wenn Kennzeichen keinen Auftragstypen hat, Dummy für die Anzeige erstellen
            if ($orderTagOrderType == null) {
                $orderTagOrderType = new OrderTypes();
                $orderTagOrderType->id = NULL;
                $orderTagOrderType->label = "Keine Zuordnung";
            }
        }

        # Alle Auftragstypen des Mandanten für Neuanlegen laden
        $allOrderTypes = OrderTypes::where('client_id', $loggedUser->client->id)->get();

        # Validierung
        $frontValidator = JsValidator::make($this->orderTagValidationRules, [], [], "#editOrderTag");

        # View anzeigen
        $arguments = ["orderTag" => $orderTag,
            "orderTagOrderType" => $orderTagOrderType,
            "allOrderTypes" => $allOrderTypes,
            "validator" => $frontValidator];

        # View rendern
        $editorHTML = View::make('order_conf.order_tags.edit')->with($arguments)->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }

    /**
     * Store a newly created resource
     *
     */
    public function store(Request $request) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::isClientAdmin($loggedUser) && !UserController::checkRight("createOrderTags", $loggedUser))
            return redirect()->route('index');

        # Speichern
        $trySave = $this->save($request, $loggedUser->client);
        if ($trySave instanceof OrderTags)
            return redirect()->route('orderTags.list')->with("message", "Auftragskennzeichen " . $trySave->label . " wurde erfolgreich angelegt.");
        # Wenn Speichern nicht erfolgreich, enthält $trySave eventuelle Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            return redirect()->back()->withErrors($trySave);
        }
    }

    /**
     * Update the specified resource
     *
     */
    public function update(Request $request, $orderTagId) {
        $loggedUser = User::find(Auth::user()->id);

        # Zugehörigen Mandanten bestimmen
        $clientID = $loggedUser->client->id;
        $client = Client::find($clientID);

        if (!$client)
            return redirect()->route('index')->withErrors("Kein Mandant zugeordnet.");

        # Berechtigung prüfen
        if (!UserController::isClientAdmin($loggedUser) && !UserController::checkRight("editOrderTags", $loggedUser))
            return redirect()->route('index');

        # Auftragskennzeichen lesen
        $orderTag = OrderTags::find($orderTagId);
        if ($orderTag == null) {
            return redirect()->back()->withErrors("Auftragskennzeichen nicht gefunden.");
        }

        # Speichern
        $trySave = $this->save($request, $client, $orderTag);
        if ($trySave instanceof OrderTags)
            return redirect()->route('orderTags.list')->with("message", "Auftragskennzeichen " . $trySave->label . " wurde erfolgreich gespeichert.");
        # Wenn Speichern nicht erfolgreich, enthält $trySave eventuelle Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            return redirect()->back()->withErrors($trySave);
        }
    }


    /**
     * Save the specified resource in storage.
     *
     */
    public function save(Request $request, Client $client, OrderTags $orderTag = null) {
        if (is_null($orderTag)) $orderTag = new OrderTags();

        # Input validieren
        $backValidator = Validator::make($request->all(), $this->orderTagValidationRules);

        if ($backValidator->fails()) {
            return $backValidator->errors();
        }

        $orderTag->client_id = $client->id;
        $orderTag->label = $request->get('label');
        $orderTag->order_types_id = $request->get('order_types_id');

        # Speichern
        try {
            $orderTag->save();
        } catch (\Exception $e) {
            Log::error("Fehler beim Speichern der Daten für Auftragskennzeichen " . $orderTag->label .
                ", (ID: " . $orderTag->id . " ), Mandant " . $client->id . ": " . $e->getMessage());
            return "Es ist ein Fehler aufgetreten.";
        }

        return $orderTag;
    }


    public function showDeleteConfirm() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("deleteOrderTags", $loggedUser))
            return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

        $orderTagId = request()->input('orderTagId');

        # Auftragskennzeichen lesen
        $orderTag = OrderTags::find($orderTagId);
        if ($orderTag == null)
            return response()->json(['response' => 'error', 'message' => 'Auftragskennzeichen nicht gefunden.']);


        $arguments = ["confItem" => "orderTags",
            "confItemLabel" => "Auftragskennzeichen",
            "confItemLabelArticle" => "das",
            "item" => $orderTag];

        # View rendern
        $editorHTML = View::make('order_conf.delete')->with($arguments)->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }


    /**
     * Remove the specified resource from storage.
     *
     */
    public function destroy() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("deleteOrderTags", $loggedUser))
            return redirect()->route('index');

        $orderTagId = request()->input('orderTagsId');
        $confirmDelete = request()->input('confirmDelete');

        # Auftragskennzeichen lesen
        $orderTag = OrderTags::find($orderTagId);
        if ($orderTag == null)
            return redirect()->back()->withErrors("Auftragskennzeichen nicht gefunden.");


        # Prüfen, ob Checkbox bestätigt ist
        if (!is_null($confirmDelete)) {
            # Auftragskennzeichen löschen - wenn möglich!
            try {
                # Versuchen, zu löschen
                $orderTag->delete();
            } catch (\Exception $e) {
                # Es kann sein, dass noch Relations zu Aufträgen (orders) bestehen
                Log::error("Fehler beim Löschen des Auftragskennzeichens " . $orderTag->id . ", Mandant " .
                    $loggedUser->client->id . ": " . $e->getMessage());

                return redirect()->back()->withErrors("Auftragskennzeichen konnte nicht gelöscht werden. Es 
            bestehen noch Verbindungen zu Aufträgen.");
            }
        }
        else {
            return redirect()->back()->with("message", "Das Auftragskennzeichen wurde nicht gelöscht, da das Löschen bestätigt werden muss!");
        }

        Log::info("Auftragskennzeichen " . $orderTag->id . ", Mandant " . $loggedUser->client->id . " wurde gelöscht. 
                            (User " . $loggedUser->id . ")");
        return redirect()->route('orderTags.list')->with("message", "Auftragskennzeichen wurde erfolgreich gelöscht.");
    }
}
