<?php

namespace App\Http\Controllers;

use App\OrderDocuments;
use App\OrderStatus;
use App\OrderTypes;
use Illuminate\Http\Request;

use App\User;
use App\Client;
use App\ProjectType;
use App\Document;
use App\DocumentRevision;
use App\DocumentTemplate;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use JsValidator;

use App\Repositories\ProjectTypeDocumentRepository;

class ProjectTypesController extends Controller {
    const PROJECTTYPE_FILES_PATH = "/files/projecttype_files/";

    private $projectTypeValidationRules = [
        'title' => 'required|string|max:255',
    ];

    /**
     * Display a listing of the resource.
     */
    public function index() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("seeProjectTypes", $loggedUser))
            return redirect()->route('index');



        # Auftragstypen des Mandanten auslesen
//        $orderTypes = $loggedUser->client->order_types;

        $projectTypes = ProjectType::all();


        return View::make('project_conf.project_types.list')->with(["projectTypes" => $projectTypes]);
    }

    public function editPage($projectTypeId){
        $loggedUser = User::find(Auth::user()->id);

//        $projectTypeId = request()->input('projectTypeId');

        $projectType = ProjectType::find($projectTypeId);
        $projectTypeTemplates = $projectType->documentTemplates;
        if ($projectType == null)
            return response()->json(['response' => 'error', 'message' => 'Projekttyp nicht gefunden.']);

        # Validierung
        $frontValidator = JsValidator::make($this->projectTypeValidationRules, [], [], "#editProjectType");

        $arguments = ["projectType" => $projectType,
            "projectTypeProjectDocumentList" => $projectTypeTemplates->where("type", "=", "project")->pluck("document"),
            "projectTypeArticleDocumentList" => $projectTypeTemplates->where("type", "=", "article")->pluck("document"),
            "validator" => $frontValidator,
//            "orderStatusList" => $orderStatusList
        ];

        return View::make('project_conf.project_types.editPage')->with($arguments)->render();
    }

    /**
     * Show the form for creating a new resource.
     * Show the form for editing the specified resource.
     *
     */
    public function edit() {
        $loggedUser = User::find(Auth::user()->id);

        $projectTypeId = request()->input('projectTypeId');

        # Neuen Auftragstyp erstellen
        if ($projectTypeId == 0) {
            # Berechtigung prüfen
            if (!UserController::checkRight("createProjectTypes", $loggedUser))
                return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

            $projectType = new ProjectType();
        } # Auftragstyp bearbeiten
        else {
            # Berechtigung prüfen
            if (!UserController::checkRight("editProjectTypes", $loggedUser))
                return response()->json(['response' => 'error', 'message' => ' Keine Berechtigung.']);

            # Auftragstyp finden
            $projectType = ProjectType::find($projectTypeId);
            if ($projectType == null)
                return response()->json(['response' => 'error', 'message' => 'Projekttyp nicht gefunden.']);

        }

        # Auftragsstatus auslesen (für Belegung Startstatus)
//        $orderStatusList = $loggedUser->client->order_status;

        # Validierung
        $frontValidator = JsValidator::make($this->projectTypeValidationRules, [], [], "#editProjectType");

        $arguments = ["projectType" => $projectType,
            "validator" => $frontValidator,
//            "orderStatusList" => $orderStatusList
        ];

        # View rendern
        $editorHTML = View::make('project_conf.project_types.edit')->with($arguments)->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }


    /**
     * Store a newly created resource
     *
     */
    public function store(Request $request) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("createProjectTypes", $loggedUser))
            return redirect()->route('index');

        # Speichern
        $trySave = $this->save($request, $loggedUser->client);
        if ($trySave instanceof ProjectType)
            return redirect()->route('projectTypes.list')->with("message", "Projekttyp " . $trySave->title . " wurde erfolgreich angelegt.");
        # Wenn Speichern nicht erfolgreich, enthält $trySave eventuelle Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            return redirect()->back()->withErrors($trySave);
        }
    }

    public function downloadFile(Request $request){
        $loggedUser = User::find(Auth::user()->id);

        
    }

    /**
     * Update the specified resource
     *
     */
    public function update(Request $request) {
        $loggedUser = User::find(Auth::user()->id);

//        $saveProjectError = false;
//        $projectTypeFilesError = false;

        ### Auftragsdokumente zuordnen ###
        $projectTypeDocumentRepository = new ProjectTypeDocumentRepository();
        $projectTypeDocumentRepository->dropzoneName = "newProjectFiles";
        $projectTypeDocumentRepository->setTemplateType("project");
        $projectTypeDocumentRepository->resolveDocuments($loggedUser, $request);
        $projectType = $projectTypeDocumentRepository->projectType;

        $projectTypeDocumentRepository = new ProjectTypeDocumentRepository();
        $projectTypeDocumentRepository->dropzoneName = "newArticleFiles";
        $projectTypeDocumentRepository->setTemplateType("article");
        $projectTypeDocumentRepository->resolveDocuments($loggedUser, $request);
        $projectType = $projectTypeDocumentRepository->projectType;


        # Speichern
        $trySave = $this->save($request, $loggedUser->client, $projectType);
        if ($trySave instanceof ProjectType)
            return redirect()->route('projectTypes.list')->with("message", "Projekttyp " . $projectType->title . " wurde erfolgreich gespeichert.");
        # Wenn Speichern nicht erfolgreich, enthält $trySave eventuelle Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            $saveProjectError = true;
            return redirect()->back()->withErrors($trySave);
        }

    }


    /**
     * Save the specified resource in storage.
     *
     */
    public function save(Request $request, Client $client, ProjectType $projectType = null) {
        if (is_null($projectType)) $projectType = new ProjectType();

        # Input validieren
        $backValidator = Validator::make($request->all(), $this->projectTypeValidationRules);

        if ($backValidator->fails()) {
            return $backValidator->errors();
        }

        # Neue Daten setzen
//        $orderType->client_id = $client->id;
        $projectType->title = $request->get('title');
//        $orderType->order_statuses_id = $request->get('order_statuses_id');

        # Speichern
        try {
            $projectType->save();
        } catch (\Exception $e) {
            Log::error("Fehler beim Speichern der Daten für Projekttyp " . $projectType->title . ", (ID: " . $projectType->id . " ), Mandant " .
                $client->id . ": " . $e->getMessage());
            return "Es ist ein Fehler aufgetreten.";
        }

        return $projectType;
    }


    public function showDeleteConfirm() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("deleteProjectTypes", $loggedUser))
            return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

        $projectTypeId = request()->input('projectTypeId');

        # Auftragstyp lesen
        $projectType = ProjectType::find($projectTypeId);
        if ($projectType == null)
            return response()->json(['response' => 'error', 'message' => 'Projekttyp nicht gefunden.']);

        $arguments = ["confItem" => "projectTypes",
            "confItemLabel" => "Projekttyp",
            "confItemLabelArticle" => "den",
            "additional_info" => "Zugehörige Projekte werden hiermit auch gelöscht!",
            "item" => $projectType];

        # View rendern
        $editorHTML = View::make('project_conf.delete')->with($arguments)->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }

    /**
     * Remove the specified resource from storage.
     *
     */
    public function destroy() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::isClientAdmin($loggedUser) && !UserController::checkRight("deleteProjectTypes", $loggedUser))
            return redirect()->route('index');

        $projectTypeId = request()->input('projectTypesId');
        $confirmDelete = request()->input('confirmDelete');

        # Auftragstyp lesen
        $projectType = ProjectType::find($projectTypeId);
        if ($projectType == null)
            return redirect()->back()->withErrors("Projekttyp nicht gefunden.");

        # Prüfen, ob Checkbox bestätigt ist
        if (!is_null($confirmDelete)) {
            # Auftragstyp löschen - wenn möglich!
            try {
                # Versuchen, zu löschen
                $projectType->delete();
            } catch (\Exception $e) {
                # Es kann sein, dass noch Relations zu Auftragskennzeichen (order_tags) oder Aufträgen (orders) bestehen
                Log::error("Fehler beim Löschen des Projekttyps " . $projectType->id . ", Mandant " .
                    $loggedUser->client->id . ": " . $e->getMessage());

                return redirect()->back()->withErrors("Projekttyp konnte nicht gelöscht werden. Es bestehen noch 
            Verbindungen zu Projekten.");
            }
        }
        else {
            return redirect()->back()->with("message", "Der Projekttyp wurde nicht gelöscht, da das Löschen bestätigt werden muss!");
        }

        Log::info("Projekttyp " . $projectType->id . ", Mandant " . $loggedUser->client->id . " wurde gelöscht. 
                            (User " . $loggedUser->id . ")");
        return redirect()->route('projectTypes.list')->with("message", "Projekttyp wurde erfolgreich gelöscht.");
    }
}
