<?php

namespace App\Http\Controllers;

use App\Rights;
use App\RightsGroup;
use App\User;
use Illuminate\Http\Request;
use Exception;
use Illuminate\Support\Facades\Log;

class RightsController extends Controller {
    protected $userRights = array();
    protected $user;


    public function __construct(User $user) {
        $this->userRights = $this->getUserRights($user);
        $this->user = $user;
    }

    private function getUserRights($user) {
        $rights = $user->right;

        # Nur interne Namen der Rechte auslesen, mehr wird nicht benötigt
        foreach ($rights as $right) {
            $this->userRights[] = $right->name;
        }
        return $this->userRights;
    }

    public function isSuperAdmin() {
        # Prüfen, ob Recht vorhanden ist
        $search = array_search('isSuperAdmin', $this->userRights);

        if ($search === false) return false;

        return true;
    }

    public function isClientAdmin() {
        if ($this->isSuperAdmin()) return true;

        # Prüfen, ob Recht vorhanden ist
        $search = array_search('isClientAdmin', $this->userRights);

        if ($search === false) return false;

        return true;
    }

    public function checkRight($right) {
        # Prüfen, ob Admin-Recht vorhanden, dann ist alles erlaubt
        if ($this->isClientAdmin()) return true;


        # Prüfen, ob Recht vorhanden ist
        $search = array_search($right, $this->userRights);

        if ($search === false) return false;

        return true;
    }

    public function removeAllRights() {
        $this->user->right()->detach();
    }

    public function removeAllSingleRights() {
        $isSuperAdmin = false;
        $isClientAdmin = false;

        # Merken, ob User globaler Administrator oder Mandantenadministrator war
        if ($this->isSuperAdmin()) $isSuperAdmin = true;
        if ($this->isClientAdmin()) $isClientAdmin = true;

        # Alle Rechte entfernen
        $this->user->right()->detach();

        # Globaler Administrator / Mandantenadministrator ggf. wieder hinzufügen
        if ($isSuperAdmin) {
            $this->makeSuperAdmin();
        }
        if ($isClientAdmin) {
            $this->makeClientAdmin();
        }
    }

    public function addRights(Array $rights) {
        try {
            $this->user->right()->syncWithoutDetaching($rights);
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return false;
        }
        return true;
    }

    public function makeClientAdmin() {
        try {
            $this->addRights([2]);
        } catch (\Exception $e) {
            Log::error($e->getMessage());
        }
        return true;
    }

    public function makeSuperAdmin() {
        try {
            $this->addRights([1]);
        } catch (\Exception $e) {
            Log::error($e->getMessage());
        }
        return true;
    }


    public static function setStandardRights($user) {

        $standardRights = [
            9,  # Bearbeiterliste ansehen
            12, # Bearbeiterkalender ansehen
            13, # Auftraggeberliste sehen
            17, # Auftragstypen ansehen
            21, # Auftragskennzeichen ansehen
            25, # Auftragsstatus ansehen
            29, # Bezahlstatus ansehen
            33, # Aufträge sehen
            38, # Textbausteine ansehen
            44, # Nummernkreise ansehen
            45, # Produkte ansehen
        ];
        try {
            $user->right()->syncWithoutDetaching($standardRights);
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return false;
        }
        return true;

    }


    public static function isExcludingRight($right) {
        $excludingRights = [
            "seeOnlyOwnOrders", # Nur zugewiesene Aufträge sehen
            "hideSensibleOrderData", # Sensible Auftragsdaten ausblenden
        ];

        if (in_array($right, $excludingRights)) return true;
        return false;
    }
}
