<?php

namespace App\Http\Controllers;

use App\Textblock;
use Illuminate\Http\Request;

use App\User;
use App\Client;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use JsValidator;

class TextblockController extends Controller {
    private $textblockValidationRules = [
        "subject" => "string|nullable|max:255",
        "text" => "required|string"
    ];

    /**
     * Display a listing of the resource.
     *
     */
    public function index() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("seeTextblocks", $loggedUser))
            return redirect()->route('index');

        # Textbausteine des Mandanten auslesen
        $textblocks = $loggedUser->client->textblocks;

        return View::make('order_conf.textblocks.list')->with(["textblocks" => $textblocks]);

    }

    /**
     * Show the form for creating a new resource.
     * Show the form for editing the specified resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function edit($textblockId) {
        $loggedUser = User::find(Auth::user()->id);

        # Neuen Textbaustein erstellen
        if ($textblockId == 0) {
            # Berechtigung prüfen
            if (!UserController::isClientAdmin($loggedUser) && !UserController::checkRight("createTextblocks", $loggedUser))
                return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

            $textblock = new Textblock();
        }
        # Textbaustein bearbeiten
        else {
            # Berechtigung prüfen
            if (!UserController::isClientAdmin($loggedUser) && !UserController::checkRight("editTextblocks", $loggedUser))
                return response()->json(['response' => 'error', 'message' => 'Es ist ein Fehler aufgetreten: Keine Berechtigung.']);

            # Textbaustein finden
            $textblock = Textblock::find($textblockId);
            if ($textblock == null)
                return response()->json(['response' => 'error', 'message' => 'Textbaustein nicht gefunden.']);
        }

        # Validierung
        $frontValidator = JsValidator::make($this->textblockValidationRules, [], [], "#editTextblock");

        $arguments = ["textblock" => $textblock,
            "validator" => $frontValidator];

        # View rendern
        $editorHTML = View::make('order_conf.textblocks.edit')->with($arguments)->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }


    /**
     * Store a newly created resource
     *
     */
    public function store(Request $request) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("createTextblocks", $loggedUser))
            return redirect()->route('index');

        # Speichern
        $trySave = $this->save($request, $loggedUser->client);
        if ($trySave instanceof Textblock)
            return redirect()->route('textblocks.list')->with("message", "Textbaustein wurde erfolgreich angelegt.");
        # Wenn Speichern nicht erfolgreich, enthält $trySave eventuelle Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            return redirect()->back()->withErrors($trySave);
        }
    }


    /**
     * Update the specified resource
     *
     */
    public function update(Request $request, $textblockId) {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("editTextblocks", $loggedUser))
            return redirect()->route('index');

        # Textbaustein lesen
        $textblock = Textblock::find($textblockId);

        if ($textblock == null)
            return redirect()->back()->withErrors("Textbaustein nicht gefunden.");

        # Speichern
        $trySave = $this->save($request, $loggedUser->client, $textblock);
        if ($trySave instanceof Textblock)
            return redirect()->route('textblocks.list')->with("message", "Textbaustein wurde erfolgreich gespeichert.");
        # Wenn Speichern nicht erfolgreich, enthält $trySave eventuelle Fehlermeldungen, die dem Nutzer angezeigt werden
        else {
            return redirect()->back()->withErrors($trySave);
        }
    }


    /**
     * Save the specified resource in storage.
     *
     */
    public function save(Request $request, Client $client, Textblock $textblock = null) {
        if (is_null($textblock)) $textblock = new Textblock();

        # Input validieren
        $backValidator = Validator::make($request->all(), $this->textblockValidationRules);

        if ($backValidator->fails()) {
            return $backValidator->errors();
        }

        # Neue Daten setzen
        $textblock->client_id = $client->id;
        $textblock->subject = $request->get('subject');
        $textblock->text = $request->get('text');

        # leeren Betreff zulassen
        if ($textblock->subject == null) {
            $textblock->subject = "";
        }

        # Speichern
        try {
            $textblock->save();
        } catch (\Exception $e) {
            Log::error("Fehler beim Speichern der Daten für Textbaustein " . $textblock->subject . ", (ID: " . $textblock->id . " ), Mandant " .
                $client->id . ": " . $e->getMessage());
            return ["Es ist ein Fehler aufgetreten."];
        }

        return $textblock;
    }


    public function showDeleteConfirm() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("deleteTextblocks", $loggedUser))
            return response()->json(['response' => 'error', 'message' => 'Keine Berechtigung.']);

        $textblockId = request()->input('textblockId');

        # Textbaustein lesen
        $textblock = Textblock::find($textblockId);
        if ($textblock == null)
            return response()->json(['response' => 'error', 'message' => 'Textbaustein nicht gefunden.']);

        $arguments = ["confItem" => "textblocks",
            "confItemLabel" => "Textbaustein",
            "confItemLabelArticle" => "den",
            "item" => $textblock];

        # View rendern
        $editorHTML = View::make('order_conf.delete')->with($arguments)->render();

        return response()->json(['response' => 'success', 'editorHTML' => $editorHTML]);
    }

    /**
     * Remove the specified resource from storage.
     *
     */
    public function destroy() {
        $loggedUser = User::find(Auth::user()->id);

        # Berechtigung prüfen
        if (!UserController::checkRight("deleteTextblocks", $loggedUser))
            return redirect()->route('index');

        $textblockId = request()->input('textblocksId');
        $confirmDelete = request()->input('confirmDelete');

        # Textbaustein lesen
        $textblock = Textblock::find($textblockId);
        if ($textblock == null)
            return redirect()->back()->withErrors("Textbaustein nicht gefunden.");


        # Prüfen, ob Checkbox bestätigt ist
        if (!is_null($confirmDelete)) {
            # Textbaustein löschen - wenn möglich!
            try {
                # Versuchen, zu löschen
                $textblock->delete();
            } catch (\Exception $e) {
                Log::error("Fehler beim Löschen des Textbausteins " . $textblock->id . ", Mandant " .
                    $loggedUser->client->id . ": " . $e->getMessage());

                return redirect()->back()->withErrors("Textbaustein konnte nicht gelöscht werden.");
            }
        }
        else {
            return redirect()->back()->with("message", "Der Textbaustein wurde nicht gelöscht, da das Löschen bestätigt werden muss!");
        }

        Log::info("Textbaustein " . $textblock->id . ", Mandant " . $loggedUser->client->id . " wurde gelöscht. 
                            (User " . $loggedUser->id . ")");
        return redirect()->route('textblocks.list')->with("message", "Textbaustein wurde erfolgreich gelöscht.");
    }


    public static function makeText($text, $placeholders) {

        # Platzhalter trennen
        $replace = array_keys($placeholders);
        $with = array_values($placeholders);

        return str_replace($replace, $with, $text);
    }
}
