<?php

namespace App\Http\Controllers;

use App\Jobs\SendConfirmationEMail;
use App\Jobs\SendRegisterClientEMailJob;
use App\Jobs\SendResetPasswordEMailJob;
use App\Jobs\SendSetNewRegisterPasswordJob;
use App\Library\Dropzone;
use App\Mail\ConfirmNewEmail;
use App\Mail\ResetPassword;
use App\Mail\SetNewRegisterPassword;
use App\User;
use App\MailQueue;
use App\UserToken;
use App\RightsGroup;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\URL;
use View;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use JsValidator;
use Validator;


use App\Jobs\SendChangeEMailJob;
use Illuminate\Support\Facades\Mail;
use App\Mail\ChangeEMail;


class UserController extends Controller {
    const REL_FILE_PATH = "/images/userImages/";

    protected $updateValidationRules = [
        'name' => 'required|string|max:255',
        'firstname' => 'required|string|max:50',
    ];

    protected $eMailValidation = [
        'email' => 'required|string|email|max:255',
    ];

    protected $emailRegisterValidation = [
        'email' => 'required|string|email|max:255|unique:users',
    ];

    protected $eMailUpdateValidation = ['email' => 'required|string|email|max:255'];

    protected $passwordValidation = [
        'password' => 'required|string|min:6',
    ];

    protected $newMailValidation = [
        'email' => 'required|string|email|max:255',
        'newEmailConfirmed' => 'required|string|email|max:255|same:email'
    ];

    protected $newPasswordValidation = [
        'newPassword' => 'required|string|min:6',
        'newPasswordConfirm' => 'required|string|min:6|same:newPassword'
    ];

    public function getUpdateValidationRules() {
        return $this->updateValidationRules;
    }

    private function getNewMailValidationRules() {
        return $this->newMailValidation;
    }

    private function getNewPasswordValidationRules() {
        return $this->newPasswordValidation;
    }

    private function getEMailValidationRules() {
        return $this->eMailValidation;
    }

    public function getEMailUpdateValidationRules() {
        return $this->eMailUpdateValidation;
    }

    public function getRegisterValidationRules() {
        return array_merge($this->updateValidationRules,
            $this->emailRegisterValidation);
    }

    public function getRegisterByAdminValidationRules() {
        $arr = ['email_confirmation' => 'required|string|email|max:255|same:email'];
        $rules = array_merge($this->updateValidationRules, $this->eMailValidation, $arr);
        return $rules;
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\User $user
     * @return \Illuminate\Http\Response
     */
    public function edit(User $user) {
        $frontValidator = JsValidator::make($this->getUpdateValidationRules(), [], [], "#userEdit");
        return View::make('user.edit')->with(["validator" => $frontValidator, "user" => $user, "activeNavItem" => "basicInformation"]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param  \App\User $user
     * @return \Illuminate\Http\Response
     */
    public function update($id) {
        $emailToSearch = null;
        # Prüfen, ob $id = ID des eingeloggten Users
        $loggedID = Auth::user()->id;

        if ($loggedID != $id) {
            # Wenn die IDs nicht übereinstimmen, muss der eingeloggte User das Recht besitzen, andere Profile zu
            # bearbeiten!
            if (!self::checkRight("editProfiles", User::find($loggedID))) {
                return redirect()->route('index')->withErrors(["Keine Berechtigung!"]);
            } # Fall: Eingeloggter User ist berechtigt und bearbeitet anderen User
            else {
                # Validierung E-Mail-Adresse ist nicht über Validator möglich, da sie ja schon vorhanden ist. Dennoch muss
                # geprüft werden, ob sie nicht schon einmal vorkommt, wenn sie geändert wurde.

                # Inputfeld auslesen
                $emailToSearch = request()->input("email");

                # Prüfen, ob valide E-Mail-Adresse
                $mailValidator = Validator::make(request()->all(), ['email' => 'required|string|email|max:255']);

                if ($mailValidator->fails())
                    return redirect()->back()->withErrors(["Diese E-Mail-Adresse ist ungültig."]);

                # User mit dieser E-Mail-Adresse suchen
                $usersMailSearch = User::where("email", $emailToSearch)->get();

                # User prüfen - im Idealfall ist es nur einer, sonst ggf. zwei
                foreach ($usersMailSearch as $tmpUser) {

                    # Wenn User mit E-Mail-Adresse gleich dem User, der bearbeitet wurde, ist alles ok, sonst Fehler!
                    if ($tmpUser->id != $id) {
                        $emailToSearch = false;
                        return redirect()->back()->withErrors(["Diese E-Mail-Adresse ist bereits in Verwendung."]);
                    }
                    else continue;
                }
            }
        }

        # User-Objekt auslesen
        $user = User::find($id);

        # Validator erstellen und Inputdaten validieren
        $validationRules = $this->getUpdateValidationRules();
        $backValidator = Validator::make(request()->all(), $validationRules);

        if ($backValidator->fails()) {
            return redirect()->back()->withErrors($backValidator->errors());
        }

        # Daten aus Formular in Userobjekt speichern
        $user = $this->putRequestDataToUser($user);

        $user = $this->setImageToObject($user);

        # ggf. neue E-Mail-Adresse setzen
        if ($emailToSearch != false) {
            $user->email = $emailToSearch;
        }

        try {
            $user->save();

        } catch (\Exception $e) {
            Log::error("Fehler beim Speichern von UserModel.");
            Log::error($e->getMessage());
            return redirect()->back()->withErrors(["Es ist ein Fehler aufgetreten."]);
        }

        return redirect()->back()->with("message", "Die Daten wurden gespeichert.");
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\User $user
     * @return \Illuminate\Http\Response
     */
    public function destroy(User $user) {
        //
    }

    private function setImageToObject($user) {
        # Bildrelevante Parameter
        $newImage = request()->input('newFile');

        # Pfad für User-Bilder
        $filePath = UserController::REL_FILE_PATH;

        # Bild löschen, wenn gewünscht
        if (request()->input('deleteFile')) {
            $this->deleteImage($user);
        }

        # Prüfen, ob neues Bild via Dropzone hochgeladen wurde
        if ($newImage != "") {
            # Altes Bild löschen
            $this->deleteImage($user);

            # Neues Bild setzen
            # Pfad hinzufügen
            $newImage = DropzoneController::DROPZONE_TMP_PATH . $newImage;
            $this->setNewImage($user, $newImage, $filePath);
        }

        return $user;
    }

    private function deleteImage($user) {
        // Prüfen, ob ein Bild für den Nutzer existiert
        if ($user->image != null) {
            $userImage = $user->image;
            if (!empty($userImage)) {
                Storage::disk('uploads')->delete($userImage);
            }
        }
        $user->image = NULL;

        return $user;
    }

    # Speichern eines neuen Bildes für den User
    private function setNewImage($user, $newImage, $filePath) {

        # Neue Dateiendung auslesen
        $info = pathinfo(Storage::disk('uploads')->url($newImage));

        if (is_array($info)) {
            $newExtension = $info["extension"];
        }
        else return $user;

        # Pfad zum temporären Bild setzen
        $newImage = Storage::disk("uploads")->getAdapter()->getPathPrefix() . $newImage;

        # Neuen Dateinamen zusammensetzen
        $filename = "userImage_" . $user->id . uniqid() . "." . $newExtension;

        # Neues Bild hochladen
        if (UploadController::uploadFile($newImage, $filename, $filePath)) {
            $user->image = $filePath . $filename;
        }

        return $user;
    }

    public function putRequestDataToUser(User $user) {
        $user = User::find($user->id);

        # Übergebene Daten aus Formular den Attributen des User-Objekts zuweisen
        $user->name = request()->input('name');
        $user->firstname = request()->input('firstname');

        return $user;
    }


    /* Handling E-Mail ändern */
    public function changeEmail() {
        $frontValidator = JsValidator::make($this->getNewMailValidationRules(), [], [], "#changeEmail");
        return View::make('user.changeEmail')->with(["validator" => $frontValidator, "activeNavItem" => "basicInformation"]);
    }

    /* E-Mail an alte E-Mail-Adresse zum Bestätigen der Änderung senden */
    public function changeEmailSent(Request $request) {
        $user = User::find(Auth::user()->id);

        # Validator erstellen und Inputdaten validieren
        $backValidator = Validator::make(request()->all(), $this->getNewMailValidationRules());

        if ($backValidator->fails()) {
            return redirect()->route('user.edit', ["activeNavItem" => "basicInformation"])->withErrors($backValidator->errors());
        }

        $newMail = request()->input('email');
        $newMailConfirmed = request()->input('newEmailConfirmed');

        # 2019-01-24 / DS / Prüfung, ob E-Mail-Adresse bereits vergeben, nur auf aktive User
        if (!UserController::checkEmailAvailable($newMail)) {
            return redirect()->back()->withErrors(["Diese E-Mail-Adresse ist bereits in Verwendung."]);
        }

        # Neue E-Mail-Adresse in User-Tabelle hinterlegen
        $user->newEmail = $newMail;
        $user->save();

        # Token für Link generieren
        $token = UserTokenController::generateToken();

        # Eintrag in Token-Tabelle schreiben
        $userTokenEntry = new UserToken();

        $userTokenEntry->userID = $user->id;
        $userTokenEntry->type = "mailUpdate";
        $userTokenEntry->token = UserTokenController::hashToken($token);

        $userTokenEntry->save();

        # E-Mail an _alte_ Mailadresse zum Bestätigen der Änderung in Queue hinterlegen
        $changeLink = URL::asset("/user/edit/changeEmailCheckConfirmation/" . $token);

        $confirmationMail = new ChangeEMail($user, $changeLink);

        try {
            $job = new SendChangeEMailJob($user->getEmailForPasswordReset(), $confirmationMail);
            dispatch($job)->onQueue('emails');
        } catch (\Exception $e) {
            # Neue E-Mail-Adresse entfernen
            $user->newEmail = NULL;
            $user->save();

            Log::error("Fehler beim Versenden der Bestätigungsmail für das E-Mail-Adresse ändern.");
            Log::error($e->getMessage());
            return redirect()->route('user.edit', ["activeNavItem" => "basicInformation"])->withErrors(["Es ist ein Fehler aufgetreten."]);
        }

        return redirect()->route('user.edit')->with("message", "Die E-Mail zur Bestätigung zum Ändern der Adresse wurde versandt.");
    }

    /* Neue E-Mail-Adresse mit weiterer Mail bestätigen */
    public function changeEmailCheckConfirmation($token) {
        $user = User::find(Auth::user()->id);

        # Prüfen, ob Token noch gültig ist
        $tokenEntry = UserTokenController::checkTokenValidity("mailUpdate", $token);
        if (!$tokenEntry) {
            $timeValid = config('constants.user.tokenValidity.mailUpdate');
            $timeValid = round((($timeValid / 60) / 60), 1);
            return redirect()->route('user.edit')->withErrors(["Der Link ist nur für " . $timeValid . " Stunden gültig. Bitte fordere einen neuen Link an."]);
        }

        # Altes Token löschen
        $tokenEntry->delete();

        # Bestätigungsnachricht an neue E-Mail-Adresse senden #

        # Neues Token generieren
        $token = UserTokenController::generateToken();

        # Eintrag in Token-Tabelle schreiben
        $userTokenEntry = new UserToken();

        $userTokenEntry->userID = $user->id;
        $userTokenEntry->type = "mailCheck";
        $userTokenEntry->token = UserTokenController::hashToken($token);
        $userTokenEntry->save();

        # E-Mail zur Bestätigung der neuen E-Mail-Adresse in Queue hinterlegen
        $confirmLink = URL::asset("/user/edit/changeEmailChanged/" . $token);

        $confirmationMail = new ConfirmNewEmail($user, $confirmLink);

        try {
            $job = new SendConfirmationEMail($user->getNewMail(), $confirmationMail);
            dispatch($job)->onQueue('emails');
        } catch (\Exception $e) {
            # Neue E-Mail-Adresse entfernen
            $user->newEmail = NULL;
            $user->save();

            Log::error("Fehler beim Versenden der E-Mail für das Bestätigen einer neuen E-Mail-Adresse.");
            Log::error($e->getMessage());
            return redirect()->route('user.edit')->withErrors(["Es ist ein Fehler aufgetreten."]);
        }

        return redirect()->route('user.edit')->with("message", "Bitte bestätige nun noch deine neue E-Mail-Adresse mit dem Link, den wir gerade versendet haben.");
    }

    /* E-Mail-Adresse endgültig auf neue Adresse ändern */
    public function changeEmailChanged($token) {
        $user = User::find(Auth::user()->id);

        # Prüfen, ob Token noch gültig ist
        $tokenEntry = UserTokenController::checkTokenValidity("mailCheck", $token);
        if (!$tokenEntry) {
            $timeValid = config('constants.user.tokenValidity.mailCheck');
            $timeValid = round((($timeValid / 60) / 60), 1);
            return redirect()->route('user.edit')->withErrors(["Der Link ist nur für " . $timeValid . " Stunden gültig. Bitte fordere einen neuen Link an."]);
        }

        # Wenn Token gültig, E-Mail aktualisieren
        $newMail = $user->newEmail;
        $user->email = $newMail;
        $user->newEmail = NULL;

        $tokenEntry->delete();

        if ($user->save())
            return redirect()->route('user.edit')->with("message", "Die E-Mail-Adresse wurde erfolgreich auf " . $newMail . " geändert.");
        else
            return redirect()->route('user.edit')->withErrors(["Es ist ein Fehler aufgetreten. Bitte fordere einen neuen Link an."]);
    }


    /* Unterseite: Sicherheit */
    public function userSecurity() {
        return View::make('user.userSecurity')->with(["activeNavItem" => "userSecurity"]);

    }

    /* Handling Passwort zurücksetzen */
    public function callPasswordReset() {
        $user = User::find(Auth::user()->id);

        $resetPassword = $this->sendPasswordResetLink($user);

        if (!$resetPassword) {
            return redirect()->route('user.userSecurity', ["activeNavItem" => "userSecurity"])->withErrors(["Es ist ein Fehler aufgetreten."]);
        }

        return redirect()->route('user.userSecurity', ["activeNavItem" => "userSecurity"])->with("message", "Der Link zum Zurücksetzen deines Passworts wurde versendet.");
    }

    public function sendPasswordResetLink(User $user, $newRegister = false) {
        # Token-Eintrag für Passwort zurücksetzen erstellen
        # Neues Token generieren
        $token = UserTokenController::generateToken();

        # Eintrag in Token-Tabelle schreiben
        $userTokenEntry = new UserToken();

        $userTokenEntry->userID = $user->id;
        $userTokenEntry->type = "resetPassword";
        $userTokenEntry->token = UserTokenController::hashToken($token);

        $userTokenEntry->save();

        # E-Mail zur Bestätigung des neuen Passworts in Queue hinterlegen
        $resetLink = URL::asset("/user/edit/passwordReset/" . $token);

        try {
            # Wenn der Nutzer neu registriert wurde, sieht die E-Mail etwas anders aus - das Procedere ist ansonsten
            # das gleiche
            if ($newRegister) {
                $mail = new SetNewRegisterPassword($user, $resetLink);
                $job = new SendSetNewRegisterPasswordJob($user->getEmailForPasswordReset(), $mail);
            }
            else {
                $mail = new ResetPassword($user, $resetLink);
                $job = new SendResetPasswordEMailJob($user->getEmailForPasswordReset(), $mail);
            }
            dispatch($job)->onQueue('emails');
        } catch (\Exception $e) {
            Log::error("Fehler beim Versenden der E-Mail für das Passwort zurücksetzen.");
            Log::error($e->getMessage());
            return false;
        }

        return true;
    }

    public function resetPassword($token) {
        # Prüfen, ob Token noch gültig ist
        $tokenEntry = UserTokenController::checkTokenValidity("resetPassword", $token);
        if (!$tokenEntry) {
            $timeValid = config('constants.user.tokenValidity.resetPassword');
            $timeValid = round((($timeValid / 60) / 60), 1);
            return redirect()->route('user.userSecurity')->withErrors(["Der Link ist nur für " . $timeValid . " Stunden gültig. Bitte fordere einen neuen Link an."]);
        }

        $tokenEntry->save();

        # Validierung hinterlegen
        $formValidator = JsValidator::make($this->getNewPasswordValidationRules(), [], [], "#userResetPassword");
        return view("user.resetPassword")->with(["token" => $token, "validator" => $formValidator, "activeNavItem" => "userSecurity"]);
    }

    public function handlePasswordReset(Request $request) {
        $backValidator = Validator::make(request()->all(), $this->getNewPasswordValidationRules());

        if ($backValidator->fails()) {
            return redirect()->route('user.userSecurity', ["activeNavItem" => "userSecurity"])->withErrors($backValidator->errors());
        }

        # Prüfen, ob Token noch gültig ist
        $tokenEntry = UserTokenController::checkTokenValidity("resetPassword", Request::get('token'));
        if (!$tokenEntry)
            return redirect()->route('user.userSecurity')->withErrors(["Ändern des Passworts fehlgeschlagen."]);

        $userToEdit = User::find($tokenEntry->userID);

        # Wenn Token und User-Bezug gültig, neues Passwort hashen und setzen
        $userToEdit->password = Hash::make(Request::get('newPassword'));

        $userToEdit->setRememberToken(Str::random(60));

        $userToEdit->active = true;

        $userToEdit->save();

        $tokenEntry->delete();

        return redirect()->route('index')->with("message", "Das neue Passwort wurde gesetzt.");
    }


    # Rechteverwaltung
    private static function getUserRights($user) {
        $rights = $user->right;
        $userRights = array();

        # Nur interne Namen der Rechte auslesen, mehr wird nicht benötigt
        foreach ($rights as $right) {
            $userRights[] = $right->name;
        }
        return $userRights;
    }

    public static function isSuperAdmin(User $user = null)
    {
        if ($user == null)
        {
            $user = User::find(Auth::user()->id);
        }

        # Prüfen, ob Recht vorhanden ist
        $search = array_search('isSuperAdmin', self::getUserRights($user));

        if ($search === false) return false;

        return true;
    }

    public static function isClientAdmin(User $user = null)
    {
        if ($user == null)
        {
            $user = User::find(Auth::user()->id);
        }

        # Prüfen, ob Super-Admin-Recht vorhanden, dann ist alles erlaubt
        if (self::isSuperAdmin($user)) return true;

        $search = array_search('isClientAdmin', self::getUserRights($user));

        if ($search === false) return false;

        return true;
    }


    public static function checkRight($right, User $user = null)
    {
        if ($user == null)
        {
            $user = User::find(Auth::user()->id);
        }

        # Sonderfall: Adminrechte abfangen
        if ($right == "isSuperAdmin") {
            if (self::isSuperAdmin($user)) return true;
            else return false;
        }

        # Sonderfall: Ausschließende Rechte, zum Beispiel "Darf nur zugewiesene Aufträge sehen"
        if (RightsController::isExcludingRight($right) && self::isClientAdmin($user)) return false;

        # Prüfen, ob Admin-Recht vorhanden, dann ist alles erlaubt
        if (self::isClientAdmin($user)) return true;

        # Prüfen, ob Recht vorhanden ist
        $search = array_search($right, self::getUserRights($user));

        if ($search === false) return false;

        return true;
    }



    # Muss der Menüpunkt "Konfiguration" angezeigt werden?
    # Prüft, ob der Nutzer überhaupt Rechte besitzt, die im Navigationsdropdown "Konfiguration" angezeigt werden
    # können (Allgemein, Auftragsverwaltung, Artikelverwaltung, E-Mails).
    # Geprüft wird lediglich auf die Rechte des Ansehens (see*), da sonst andere Rechte (edit*, create* etc.) ohnehin
    # keinen Sinn hätten
    public static function hasAnyConfigurationRights($user) {

        if (self::hasAnyGeneralConfigurationRights($user)) return true;
        if (self::hasAnyOrderConfigurationRights($user)) return true;
        if (self::hasAnyProductConfigurationRights($user)) return true;
        if (self::hasAnyMailConfigurationRights($user)) return true;

        return false;

    }

    public static function hasAnyGeneralConfigurationRights($user) {

        # Keine eigentliche Rechtegruppe, sondern nur ein Abschnitt in der Navigation "Allgemein"
        $rights = [
            "seeNumberRanges" # Nummernkreise ansehen
        ];

        # Für jedes Recht prüfen, ob der Nutzer dieses besitzt
        foreach ($rights as $right) {
            if (self::checkRight($right, $user)) {
                return true;
            }
        }

        return false;
    }

    public static function hasAnyOrderConfigurationRights($user) {
        $rightsGroup = RightsGroup::find(5);

        # Für jedes Recht prüfen, ob der Nutzer dieses besitzt
        foreach ($rightsGroup->rights as $right) {
            if (self::checkRight($right->name, $user)) {
                return true;
            }
        }

        return false;
    }

    public static function hasAnyProductConfigurationRights($user) {

        # Keine eigentliche Rechtegruppe, sondern nur ein Abschnitt in der Navigation "Produkte"
        $rights = [
            "seeProducts", # Artikel ansehen
        ];

        # Für jedes Recht prüfen, ob der Nutzer dieses besitzt
        foreach ($rights as $right) {
            if (self::checkRight($right, $user)) {
                return true;
            }
        }

        return false;
    }

    public static function hasAnyMailConfigurationRights($user) {

        # Keine eigentliche Rechtegruppe, sondern nur ein Abschnitt in der Navigation "E-Mail"
        $rights = [
            "seeTextblocks", # Textbausteine ansehen
        ];

        # Für jedes Recht prüfen, ob der Nutzer dieses besitzt
        foreach ($rights as $right) {
            if (self::checkRight($right, $user)) {
                return true;
            }
        }

        return false;
    }


    # Konvertiert die auftragsbezogenen Rechte (Editieren, Anlegen, Löschen) für die Listenansicht zu JSON-parsefähigen
    # Werten
    public static function convertOrderRightsToJs($user) {
        # Nutzerrechte fürs Parsen in JS abfragen
        if (UserController::checkRight("editOrders", $user)) $rights["canEditOrders"] = "1";
        else    $rights["canEditOrders"] = 0;
        if (UserController::checkRight("deleteOrders", $user)) $rights["canDeleteOrders"] = "1";
        else    $rights["canDeleteOrders"] = 0;
        if (UserController::checkRight("createOrders", $user)) $rights["canCreateOrders"] = "1";
        else    $rights["canCreateOrders"] = 0;
        if (UserController::checkRight("hideSensibleOrderData", $user)) $rights["hideSensibleOrderData"] = "1";
        else    $rights["hideSensibleOrderData"] = 0;

        return $rights;
    }


    public static function isResponsibleOperatorForOrder($operator, $order) {

        # Verantwortliche des Auftrags auslesen
        $responsibleOperators = $order->responsibleOperators;

        # Prüfen, ob User-Bearbeiter unter ihnen ist
        if ($responsibleOperators->contains('id', $operator->id))
            return true;
        return false;
    }


    # Prüfen, ob E-Mail-Adresse im System noch verfügbar ist
    public static function checkEmailAvailable($eMailToCheck) {

        $users = User::where("email", "=", $eMailToCheck)->first();

        # User mit dieser E-Mail-Adresse gefunden
        if ($users == NULL)
            return true;

        return false;

    }
}
