<?php

namespace App;

use App\Mail\OrderMail;
use App\OrderLogs;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Carbon\Carbon;

class Orders extends Model {
    use \Askedio\SoftCascade\Traits\SoftCascadeTrait;

    use SoftDeletes;

    protected $softCascade = ['followUps', 'positions', 'creditNotes'];

    protected $dates = ['deleted_at'];

    # Mandant
    public function client() {
        return $this->belongsTo(Client::class);
    }

    # Auftraggeber
    public function customer() {
        return $this->belongsTo(Customer::class);
    }

    # Auftragstyp
    public function order_type() {
        return $this->belongsTo(OrderTypes::class);
    }

    # Auftragsstatus
    public function order_status() {
        return $this->belongsTo(OrderStatus::class);
    }

    # Bezahlstatus
    public function order_payment_status() {
        return $this->belongsTo(OrderPaymentStatus::class);
    }

    # Auftragskennzeichen mit Wert
    public function order_tags() {
        return $this->belongsToMany(OrderTags::class, 'order_order_tag')->withPivot('tag_value');
    }

    # Bearbeiter mit Termin
    public function operators() {
        return $this->belongsToMany(Operator::class, 'order_appointment_operator')->withPivot('id', 'appointment_from', 'appointment_to');
    }

    # Verantwortlicher
    public function responsibleOperators() {
        return $this->belongsToMany(Operator::class, 'order_operators', 'order_id')->orderBy('name');
    }

    # Adressen
    public function addresses() {
        return $this->hasMany(OrderAddresses::class, 'order_id');
    }

    # Auftragsdokumente
    public function documents() {
        return $this->hasMany(OrderDocuments::class, 'order_id');
    }

    # Auftragshistorie
    public function order_logs() {
        return $this->hasMany(OrderLogs::class, 'order_id');
    }

    # Auftragshistorie - nur Freitextlognachrichten von Nutzern
    public function order_custom_logs() {
        return $this->hasMany(OrderLogs::class, 'order_id')->where('type', '=', OrderLogs::LOGTYPE_USRLOG);
    }

    # Auftragsmails
    public function mails() {
        return $this->hasMany(OrderMailLog::class, 'order_id');
    }

    # Wiedervorlagen
    public function followUpsUser($id) {
        return $this->hasMany(OrderFollowUp::class, 'order_id')->where('operator_id', '=', $id)->get();
    }

    public function followUps() {
        return $this->hasMany(OrderFollowUp::class, 'order_id');
    }

    # Rechnungen
    public function order_invoices() {
        return $this->hasMany(OrderInvoice::class, 'order_id');
    }

    # Positionen
    public function positions() {
        return $this->hasMany(OrderPositions::class, 'order_id');
    }

    # Auftragswert
    public function getOrderValue() {
        $rel = $this->hasMany(OrderPositions::class, 'order_id')->where('position_text', '=', 'Auftragswert')->pluck('position_value');

        if ($rel <> null && count($rel->all()) > 0)
            return $rel->all()[0];
        else
            return 0;
    }

    # Auftragswert für Anzeige, also mit Komma-Dezimal
    public function getOrderValueDisplay() {
        $orderValue = $this->getOrderValue();

        return (str_replace('.', ',', $orderValue));
    }

    # Enddatum Auftrag für HTML-Date formatieren
    public function getEndDateDateString() {
        $c = Carbon::createFromFormat("Y-m-d H:i:s", $this->end_date);
        return $c->toDateString();
    }

    # Enddatum Auftrag für Anzeige formatieren
    public function getReadableEndDate() {
        $c = Carbon::createFromFormat("Y-m-d H:i:s", $this->end_date);
        return $c->format("d.m.Y");
    }

    # Enddatum Auftrag für HTML-Date formatieren
    public function getExportDateDateString() {
        $c = Carbon::createFromFormat("Y-m-d H:i:s", $this->export_date);
        return $c->toDateString();
    }

    # Erstellt-Datum Auftrag für Anzeige formatieren
    public function getReadableCreated() {
        $c = Carbon::createFromFormat("Y-m-d H:i:s", $this->created);
        return $c->format("d.m.Y");
    }

    # Gutschriften
    public function creditNotes() {
        return $this->hasMany(OrderCreditNote::class, 'order_id');
    }

}
